/*
 * Player Plugin for jQuery JavaScript Library
 * http://www.jplayer.org
 *
 * Copyright (c) 2009 - 2014 Happyworm Ltd
 * Licensed under the MIT license.
 * http://opensource.org/licenses/MIT
 *
 * Author: Mark J Panaghiston
 * Version: 2.9.2
 * Date: 14th December 2014
 */

/* Support for Zepto 1.0 compiled with optional data module.
 * For AMD or NODE/CommonJS support, you will need to manually switch the related 2 lines in the code below.
 * Search terms: "jQuery Switch" and "Zepto Switch"
 */

(function (root, factory) {
	if (typeof define === 'function' && define.amd) {
		// AMD. Register as an anonymous module.
		define(['jquery'], factory); // jQuery Switch
		// define(['zepto'], factory); // Zepto Switch
	} else if (typeof exports === 'object') {
		// Node/CommonJS
		factory(require('jquery')); // jQuery Switch
		//factory(require('zepto')); // Zepto Switch
	} else {
		// Browser globals
		if(root.jQuery) { // Use jQuery if available
			factory(root.jQuery);
		} else { // Otherwise, use Zepto
			factory(root.Zepto);
		}
	}
}(this, function ($, undefined) {

	// Adapted from jquery.ui.widget.js (1.8.7): $.widget.bridge - Tweaked $.data(this,XYZ) to $(this).data(XYZ) for Zepto
	$.fn.dlePlayerVideo = function( options ) {
		var name = "dlePlayerVideo";
		var isMethodCall = typeof options === "string",
			args = Array.prototype.slice.call( arguments, 1 ),
			returnValue = this;

		// allow multiple hashes to be passed on init
		options = !isMethodCall && args.length ?
			$.extend.apply( null, [ true, options ].concat(args) ) :
			options;

		// prevent calls to internal methods
		if ( isMethodCall && options.charAt( 0 ) === "_" ) {
			return returnValue;
		}

		if ( isMethodCall ) {
			this.each(function() {
				var instance = $(this).data( name ),
					methodValue = instance && $.isFunction( instance[options] ) ?
						instance[ options ].apply( instance, args ) :
						instance;
				if ( methodValue !== instance && methodValue !== undefined ) {
					returnValue = methodValue;
					return false;
				}
			});
		} else {
			this.each(function() {
				var instance = $(this).data( name );
				if ( instance ) {
					// instance.option( options || {} )._init(); // Orig jquery.ui.widget.js code: Not recommend for dlePlayerVideo. ie., Applying new options to an existing instance (via the dlePlayerVideo constructor) and performing the _init(). The _init() is what concerns me. It would leave a lot of event handlers acting on dlePlayerVideo instance and the interface.
					instance.option( options || {} ); // The new constructor only changes the options. Changing options only has basic support atm.
				} else {
					$(this).data( name, new $.dlePlayerVideo( options, this ) );
				}
			});
		}

		return returnValue;
	};

	$.dlePlayerVideo = function( options, element ) {
		// allow instantiation without initializing for simple inheritance
		if ( arguments.length ) {
			this.element = $(element);
			this.options = $.extend(true, {},
				this.options,
				options
			);
			var self = this;
			this.element.bind( "remove.dlePlayerVideo", function() {
				self.destroy();
			});
			this._init();
		}
	};
	// End of: (Adapted from jquery.ui.widget.js (1.8.7))

	// Zepto is missing one of the animation methods.
	if(typeof $.fn.stop !== 'function') {
		$.fn.stop = function() {};
	}

	// Emulated HTML5 methods and properties
	$.dlePlayerVideo.emulateMethods = "load play pause";
	$.dlePlayerVideo.emulateStatus = "src readyState networkState currentTime duration paused ended playbackRate";
	$.dlePlayerVideo.emulateOptions = "muted volume";

	// Reserved event names generated by dlePlayerVideo that are not part of the HTML5 Media element spec
	$.dlePlayerVideo.reservedEvent = "ready flashreset resize repeat error warning";

	// Events generated by dlePlayerVideo
	$.dlePlayerVideo.event = {};
	$.each(
		[
			'ready',
			'setmedia', // Fires when the media is set
			'flashreset', // Similar to the ready event if the Flash solution is set to display:none and then shown again or if it's reloaded for another reason by the browser. For example, using CSS position:fixed on Firefox for the full screen feature.
			'resize', // Occurs when the size changes through a full/restore screen operation or if the size/sizeFull options are changed.
			'repeat', // Occurs when the repeat status changes. Usually through clicks on the repeat button of the interface.
			'click', // Occurs when the user clicks on one of the following: poster image, html video, flash video.
			'error', // Event error code in event.dlePlayerVideo.error.type. See $.dlePlayerVideo.error
			'warning', // Event warning code in event.dlePlayerVideo.warning.type. See $.dlePlayerVideo.warning

			// Other events match HTML5 spec.
			'loadstart',
			'progress',
			'suspend',
			'abort',
			'emptied',
			'stalled',
			'play',
			'pause',
			'loadedmetadata',
			'loadeddata',
			'waiting',
			'playing',
			'canplay',
			'canplaythrough',
			'seeking',
			'seeked',
			'timeupdate',
			'ended',
			'ratechange',
			'durationchange',
			'volumechange'
		],
		function() {
			$.dlePlayerVideo.event[ this ] = 'dlePlayerVideo_' + this;
		}
	);

	$.dlePlayerVideo.htmlEvent = [ // These HTML events are bubbled through to the dlePlayerVideo event, without any internal action.
		"loadstart",
		// "progress", // dlePlayerVideo uses internally before bubbling.
		// "suspend", // dlePlayerVideo uses internally before bubbling.
		"abort",
		// "error", // dlePlayerVideo uses internally before bubbling.
		"emptied",
		"stalled",
		// "play", // dlePlayerVideo uses internally before bubbling.
		// "pause", // dlePlayerVideo uses internally before bubbling.
		"loadedmetadata",
		// "loadeddata", // dlePlayerVideo uses internally before bubbling.
		// "waiting", // dlePlayerVideo uses internally before bubbling.
		// "playing", // dlePlayerVideo uses internally before bubbling.
		"canplay",
		"canplaythrough"
		// "seeking", // dlePlayerVideo uses internally before bubbling.
		// "seeked", // dlePlayerVideo uses internally before bubbling.
		// "timeupdate", // dlePlayerVideo uses internally before bubbling.
		// "ended", // dlePlayerVideo uses internally before bubbling.
		// "ratechange" // dlePlayerVideo uses internally before bubbling.
		// "durationchange" // dlePlayerVideo uses internally before bubbling.
		// "volumechange" // dlePlayerVideo uses internally before bubbling.
	];

	$.dlePlayerVideo.pause = function() {
		$.dlePlayerVideo.prototype.destroyRemoved();
		$.each($.dlePlayerVideo.prototype.instances, function(i, element) {
			if(element.data("dlePlayerVideo").status.srcSet) { // Check that media is set otherwise would cause error event.
				element.dlePlayerVideo("pause");
			}
		});
	};

	// Default for dlePlayerVideo option.timeFormat
	$.dlePlayerVideo.timeFormat = {
		showHour: false,
		showMin: true,
		showSec: true,
		padHour: false,
		padMin: true,
		padSec: true,
		sepHour: ":",
		sepMin: ":",
		sepSec: ""
	};
	var ConvertTime = function() {
		this.init();
	};
	ConvertTime.prototype = {
		init: function() {
			this.options = {
				timeFormat: $.dlePlayerVideo.timeFormat
			};
		},
		time: function(s) { // function used on dlePlayerVideo.prototype._convertTime to enable per instance options.
			s = (s && typeof s === 'number') ? s : 0;

			var myTime = new Date(s * 1000),
				hour = myTime.getUTCHours(),
				min = this.options.timeFormat.showHour ? myTime.getUTCMinutes() : myTime.getUTCMinutes() + hour * 60,
				sec = this.options.timeFormat.showMin ? myTime.getUTCSeconds() : myTime.getUTCSeconds() + min * 60,
				strHour = (this.options.timeFormat.padHour && hour < 10) ? "0" + hour : hour,
				strMin = (this.options.timeFormat.padMin && min < 10) ? "0" + min : min,
				strSec = (this.options.timeFormat.padSec && sec < 10) ? "0" + sec : sec,
				strTime = "";

			strTime += this.options.timeFormat.showHour ? strHour + this.options.timeFormat.sepHour : "";
			strTime += this.options.timeFormat.showMin ? strMin + this.options.timeFormat.sepMin : "";
			strTime += this.options.timeFormat.showSec ? strSec + this.options.timeFormat.sepSec : "";

			return strTime;
		}
	};
	var myConvertTime = new ConvertTime();
	$.dlePlayerVideo.convertTime = function(s) {
		return myConvertTime.time(s);
	};

	// Adapting jQuery 1.4.4 code for jQuery.browser. Required since jQuery 1.3.2 does not detect Chrome as webkit.
	$.dlePlayerVideo.uaBrowser = function( userAgent ) {
		var ua = userAgent.toLowerCase();

		// Useragent RegExp
		var rwebkit = /(webkit)[ \/]([\w.]+)/;
		var ropera = /(opera)(?:.*version)?[ \/]([\w.]+)/;
		var rmsie = /(msie) ([\w.]+)/;
		var rmozilla = /(mozilla)(?:.*? rv:([\w.]+))?/;

		var match = rwebkit.exec( ua ) ||
			ropera.exec( ua ) ||
			rmsie.exec( ua ) ||
			ua.indexOf("compatible") < 0 && rmozilla.exec( ua ) ||
			[];

		return { browser: match[1] || "", version: match[2] || "0" };
	};

	// Platform sniffer for detecting mobile devices
	$.dlePlayerVideo.uaPlatform = function( userAgent ) {
		var ua = userAgent.toLowerCase();

		// Useragent RegExp
		var rplatform = /(ipad|iphone|ipod|android|blackberry|playbook|windows ce|webos)/;
		var rtablet = /(ipad|playbook)/;
		var randroid = /(android)/;
		var rmobile = /(mobile)/;

		var platform = rplatform.exec( ua ) || [];
		var tablet = rtablet.exec( ua ) ||
			!rmobile.exec( ua ) && randroid.exec( ua ) ||
			[];

		if(platform[1]) {
			platform[1] = platform[1].replace(/\s/g, "_"); // Change whitespace to underscore. Enables dot notation.
		}

		return { platform: platform[1] || "", tablet: tablet[1] || "" };
	};

	$.dlePlayerVideo.browser = {
	};
	$.dlePlayerVideo.platform = {
	};

	var browserMatch = $.dlePlayerVideo.uaBrowser(navigator.userAgent);
	if ( browserMatch.browser ) {
		$.dlePlayerVideo.browser[ browserMatch.browser ] = true;
		$.dlePlayerVideo.browser.version = browserMatch.version;
	}
	var platformMatch = $.dlePlayerVideo.uaPlatform(navigator.userAgent);
	if ( platformMatch.platform ) {
		$.dlePlayerVideo.platform[ platformMatch.platform ] = true;
		$.dlePlayerVideo.platform.mobile = !platformMatch.tablet;
		$.dlePlayerVideo.platform.tablet = !!platformMatch.tablet;
	}

	// Internet Explorer (IE) Browser Document Mode Sniffer. Based on code at:
	// http://msdn.microsoft.com/en-us/library/cc288325%28v=vs.85%29.aspx#GetMode
	$.dlePlayerVideo.getDocMode = function() {
		var docMode;
		if ($.dlePlayerVideo.browser.msie) {
			if (document.documentMode) { // IE8 or later
				docMode = document.documentMode;
			} else { // IE 5-7
				docMode = 5; // Assume quirks mode unless proven otherwise
				if (document.compatMode) {
					if (document.compatMode === "CSS1Compat") {
						docMode = 7; // standards mode
					}
				}
			}
		}
		return docMode;
	};
	$.dlePlayerVideo.browser.documentMode = $.dlePlayerVideo.getDocMode();

	$.dlePlayerVideo.nativeFeatures = {
		init: function() {

			/* Fullscreen function naming influenced by W3C naming.
			 * No support for: Mozilla Proposal: https://wiki.mozilla.org/Gecko:FullScreenAPI
			 */

			var d = document,
				v = d.createElement('video'),
				spec = {
					// http://www.w3.org/TR/fullscreen/
					w3c: [
						'fullscreenEnabled',
						'fullscreenElement',
						'requestFullscreen',
						'exitFullscreen',
						'fullscreenchange',
						'fullscreenerror'
					],
					// https://developer.mozilla.org/en-US/docs/DOM/Using_fullscreen_mode
					moz: [
						'mozFullScreenEnabled',
						'mozFullScreenElement',
						'mozRequestFullScreen',
						'mozCancelFullScreen',
						'mozfullscreenchange',
						'mozfullscreenerror'
					],
					// http://developer.apple.com/library/safari/#documentation/WebKit/Reference/ElementClassRef/Element/Element.html
					// http://developer.apple.com/library/safari/#documentation/UserExperience/Reference/DocumentAdditionsReference/DocumentAdditions/DocumentAdditions.html
					webkit: [
						'',
						'webkitCurrentFullScreenElement',
						'webkitRequestFullScreen',
						'webkitCancelFullScreen',
						'webkitfullscreenchange',
						''
					],
					// http://developer.apple.com/library/safari/#documentation/AudioVideo/Reference/HTMLVideoElementClassReference/HTMLVideoElement/HTMLVideoElement.html
					// https://developer.apple.com/library/safari/samplecode/HTML5VideoEventFlow/Listings/events_js.html#//apple_ref/doc/uid/DTS40010085-events_js-DontLinkElementID_5
					// Events: 'webkitbeginfullscreen' and 'webkitendfullscreen'
					webkitVideo: [
						'webkitSupportsFullscreen',
						'webkitDisplayingFullscreen',
						'webkitEnterFullscreen',
						'webkitExitFullscreen',
						'',
						''
					],
					ms: [
						'',
						'msFullscreenElement',
						'msRequestFullscreen',
						'msExitFullscreen',
						'MSFullscreenChange',
						'MSFullscreenError'
					]
				},
				specOrder = [
					'w3c',
					'moz',
					'webkit',
					'webkitVideo',
					'ms'
				],
				fs, i, il;

			this.fullscreen = fs = {
				support: {
					w3c: !!d[spec.w3c[0]],
					moz: !!d[spec.moz[0]],
					webkit: typeof d[spec.webkit[3]] === 'function',
					webkitVideo: typeof v[spec.webkitVideo[2]] === 'function',
					ms: typeof v[spec.ms[2]] === 'function'
				},
				used: {}
			};

			// Store the name of the spec being used and as a handy boolean.
			for(i = 0, il = specOrder.length; i < il; i++) {
				var n = specOrder[i];
				if(fs.support[n]) {
					fs.spec = n;
					fs.used[n] = true;
					break;
				}
			}

			if(fs.spec) {
				var s = spec[fs.spec];
				fs.api = {
					fullscreenEnabled: true,
					fullscreenElement: function(elem) {
						elem = elem ? elem : d; // Video element required for webkitVideo
						return elem[s[1]];
					},
					requestFullscreen: function(elem) {
						return elem[s[2]](); // Chrome and Opera want parameter (Element.ALLOW_KEYBOARD_INPUT) but Safari fails if flag used.
					},
					exitFullscreen: function(elem) {
						elem = elem ? elem : d; // Video element required for webkitVideo
						return elem[s[3]]();
					}
				};
				fs.event = {
					fullscreenchange: s[4],
					fullscreenerror: s[5]
				};
			} else {
				fs.api = {
					fullscreenEnabled: false,
					fullscreenElement: function() {
						return null;
					},
					requestFullscreen: function() {},
					exitFullscreen: function() {}
				};
				fs.event = {};
			}
		}
	};
	$.dlePlayerVideo.nativeFeatures.init();

	// The keyboard control system.

	// The current dlePlayerVideo instance in focus.
	$.dlePlayerVideo.focus = null;

	// The list of element node names to ignore with key controls.
	$.dlePlayerVideo.keyIgnoreElementNames = "A INPUT TEXTAREA SELECT BUTTON";

	// The function that deals with key presses.
	var keyBindings = function(event) {
		var f = $.dlePlayerVideo.focus,
			ignoreKey;

		// A dlePlayerVideo instance must be in focus. ie., keyEnabled and the last one played.
		if(f) {
			// What generated the key press?
			$.each( $.dlePlayerVideo.keyIgnoreElementNames.split(/\s+/g), function(i, name) {
				// The strings should already be uppercase.
				if(event.target.nodeName.toUpperCase() === name.toUpperCase()) {
					ignoreKey = true;
					return false; // exit each.
				}
			});
			if(!ignoreKey) {
				// See if the key pressed matches any of the bindings.
				$.each(f.options.keyBindings, function(action, binding) {
					// The binding could be a null when the default has been disabled. ie., 1st clause in if()
					if(
						(binding && $.isFunction(binding.fn)) &&
						((typeof binding.key === 'number' && event.which === binding.key) ||
						(typeof binding.key === 'string' && event.key === binding.key))
					) {
						event.preventDefault(); // Key being used by dlePlayerVideo, so prevent default operation.
						binding.fn(f);
						return false; // exit each.
					}
				});
			}
		}
	};

	$.dlePlayerVideo.keys = function(en) {
		var event = "keydown.dlePlayerVideo";
		// Remove any binding, just in case enabled more than once.
		$(document.documentElement).unbind(event);
		if(en) {
			$(document.documentElement).bind(event, keyBindings);
		}
	};

	// Enable the global key control handler ready for any dlePlayerVideo instance with the keyEnabled option enabled.
	$.dlePlayerVideo.keys(true);

	$.dlePlayerVideo.prototype = {
		count: 0, // Static Variable: Change it via prototype.
		version: { // Static Object
			script: "2.9.2",
			needFlash: "2.9.0",
			flash: "unknown"
		},
		options: { // Instanced in $.dlePlayerVideo() constructor
			swfPath: "js", // Path to jquery.jplayer.swf. Can be relative, absolute or server root relative.
			solution: "html, flash", // Valid solutions: html, flash, aurora. Order defines priority. 1st is highest,
			supplied: "mp3", // Defines which formats dlePlayerVideo will try and support and the priority by the order. 1st is highest,
			auroraFormats: "wav", // List the aurora.js codecs being loaded externally. Its core supports "wav". Specify format in dlePlayerVideo context. EG., The aac.js codec gives the "m4a" format.
			preload: 'metadata',  // HTML5 Spec values: none, metadata, auto.
			volume: 0.8, // The volume. Number 0 to 1.
			muted: false,
			remainingDuration: false, // When true, the remaining time is shown in the duration GUI element.
			toggleDuration: false, // When true, clicks on the duration toggle between the duration and remaining display.
			captureDuration: true, // When true, clicks on the duration are captured and no longer propagate up the DOM.
			playbackRate: 1,
			defaultPlaybackRate: 1,
			minPlaybackRate: 0.5,
			maxPlaybackRate: 4,
			wmode: "opaque", // Valid wmode: window, transparent, opaque, direct, gpu.
			backgroundColor: "#000000", // To define the dlePlayerVideo div and Flash background color.
			cssSelectorAncestor: "#jp_video_container_1",
			cssSelector: { // * denotes properties that should only be required when video media type required. _cssSelector() would require changes to enable splitting these into Audio and Video defaults.
				videoPlay: ".mdtc-clnplrv-video-play", // *
				play: ".mdtc-clnplrv-play",
				pause: ".mdtc-clnplrv-pause",
				stop: ".mdtc-clnplrv-stop",
				seekBar: ".mdtc-clnplrv-seek-bar",
				playBar: ".mdtc-clnplrv-play-bar",
				mute: ".mdtc-clnplrv-mute",
				unmute: ".mdtc-clnplrv-unmute",
				volumeBar: ".mdtc-clnplrv-volume-bar",
				volumeBarValue: ".mdtc-clnplrv-volume-bar-value",
				volumeMax: ".mdtc-clnplrv-volume-max",
				playbackRateBar: ".mdtc-clnplrv-playback-rate-bar",
				playbackRateBarValue: ".mdtc-clnplrv-playback-rate-bar-value",
				currentTime: ".mdtc-clnplrv-current-time",
				duration: ".mdtc-clnplrv-duration",
				title: ".mdtc-clnplrv-title",
				fullScreen: ".mdtc-clnplrv-full-screen", // *
				restoreScreen: ".mdtc-clnplrv-restore-screen", // *
				repeat: ".mdtc-clnplrv-repeat",
				repeatOff: ".mdtc-clnplrv-repeat-off",
				gui: ".mdtc-clnplrv-gui", // The interface used with autohide feature.
				noSolution: ".mdtc-clnplrv-no-solution" // For error feedback when dlePlayerVideo cannot find a solution.
			},
			stateClass: { // Classes added to the cssSelectorAncestor to indicate the state.
				playing: "mdtc-clnplrv-state-playing",
				seeking: "mdtc-clnplrv-state-seeking",
				muted: "mdtc-clnplrv-state-muted",
				looped: "mdtc-clnplrv-state-looped",
				fullScreen: "mdtc-clnplrv-state-full-screen",
				noVolume: "mdtc-clnplrv-state-no-volume"
			},
			useStateClassSkin: false, // A state class skin relies on the state classes to change the visual appearance. The single control toggles the effect, for example: play then pause, mute then unmute.
			autoBlur: true, // GUI control handlers will drop focus after clicks.
			smoothPlayBar: false, // Smooths the play bar transitions, which affects clicks and short media with big changes per second.
			fullScreen: false, // Native Full Screen
			fullWindow: false,
			autohide: {
				restored: false, // Controls the interface autohide feature.
				full: true, // Controls the interface autohide feature.
				fadeIn: 200, // Milliseconds. The period of the fadeIn anim.
				fadeOut: 600, // Milliseconds. The period of the fadeOut anim.
				hold: 1000 // Milliseconds. The period of the pause before autohide beings.
			},
			loop: false,
			repeat: function(event) { // The default dlePlayerVideo repeat event handler
				if(event.dlePlayerVideo.options.loop) {
					$(this).unbind(".dlePlayerVideoRepeat").bind($.dlePlayerVideo.event.ended + ".dlePlayerVideo.dlePlayerVideoRepeat", function() {
						$(this).dlePlayerVideo("play");
					});
				} else {
					$(this).unbind(".dlePlayerVideoRepeat");
				}
			},
			nativeVideoControls: {
				// Works well on standard browsers.
				// Phone and tablet browsers can have problems with the controls disappearing.
			},
			noFullWindow: {
				msie: /msie [0-6]\./,
				ipad: /ipad.*?os [0-4]\./,
				iphone: /iphone/,
				ipod: /ipod/,
				android_pad: /android [0-3]\.(?!.*?mobile)/,
				android_phone: /(?=.*android)(?!.*chrome)(?=.*mobile)/,
				blackberry: /blackberry/,
				windows_ce: /windows ce/,
				iemobile: /iemobile/,
				webos: /webos/
			},
			noVolume: {
				ipad: /ipad/,
				iphone: /iphone/,
				ipod: /ipod/,
				android_pad: /android(?!.*?mobile)/,
				android_phone: /android.*?mobile/,
				blackberry: /blackberry/,
				windows_ce: /windows ce/,
				iemobile: /iemobile/,
				webos: /webos/,
				playbook: /playbook/
			},
			timeFormat: {
				// Specific time format for this instance. The supported options are defined in $.dlePlayerVideo.timeFormat
				// For the undefined options we use the default from $.dlePlayerVideo.timeFormat
			},
			keyEnabled: false, // Enables keyboard controls.
			audioFullScreen: false, // Enables keyboard controls to enter full screen with audio media.
			keyBindings: { // The key control object, defining the key codes and the functions to execute.
				// The parameter, f = $.dlePlayerVideo.focus, will be checked truethy before attempting to call any of these functions.
				// Properties may be added to this object, in key/fn pairs, to enable other key controls. EG, for the playlist add-on.
				play: {
					key: 80, // p
					fn: function(f) {
						if(f.status.paused) {
							f.play();
						} else {
							f.pause();
						}
					}
				},
				fullScreen: {
					key: 70, // f
					fn: function(f) {
						if(f.status.video || f.options.audioFullScreen) {
							f._setOption("fullScreen", !f.options.fullScreen);
						}
					}
				},
				muted: {
					key: 77, // m
					fn: function(f) {
						f._muted(!f.options.muted);
					}
				},
				volumeUp: {
					key: 190, // .
					fn: function(f) {
						f.volume(f.options.volume + 0.1);
					}
				},
				volumeDown: {
					key: 188, // ,
					fn: function(f) {
						f.volume(f.options.volume - 0.1);
					}
				},
				loop: {
					key: 76, // l
					fn: function(f) {
						f._loop(!f.options.loop);
					}
				}
			},
			verticalVolume: false, // Calculate volume from the bottom of the volume bar. Default is from the left. Also volume affects either width or height.
			verticalPlaybackRate: false,
			globalVolume: false, // Set to make volume and muted changes affect all dlePlayerVideo instances with this option enabled
			idPrefix: "jp_video_", // Prefix for the ids of html elements created by dlePlayerVideo. For flash, this must not include characters: . - + * / \
			noConflict: "jQuery",
			emulateHtml: false, // Emulates the HTML5 Media element on the dlePlayerVideo element.
			consoleAlerts: true, // Alerts are sent to the console.log() instead of alert().
			errorAlerts: false,
			warningAlerts: false
		},
		optionsAudio: {
			size: {
				width: "0px",
				height: "0px",
				cssClass: ""
			},
			sizeFull: {
				width: "0px",
				height: "0px",
				cssClass: ""
			}
		},
		optionsVideo: {
			size: {
				width: "480px",
				height: "270px",
				cssClass: "mdtc-clnplrv-video-270p"
			},
			sizeFull: {
				width: "100%",
				height: "100%",
				cssClass: "mdtc-clnplrv-video-full"
			}
		},
		instances: {}, // Static Object
		status: { // Instanced in _init()
			src: "",
			media: {},
			paused: true,
			format: {},
			formatType: "",
			waitForPlay: true, // Same as waitForLoad except in case where preloading.
			waitForLoad: true,
			srcSet: false,
			video: false, // True if playing a video
			seekPercent: 0,
			currentPercentRelative: 0,
			currentPercentAbsolute: 0,
			currentTime: 0,
			duration: 0,
			remaining: 0,
			videoWidth: 0, // Intrinsic width of the video in pixels.
			videoHeight: 0, // Intrinsic height of the video in pixels.
			readyState: 0,
			networkState: 0,
			playbackRate: 1, // Warning - Now both an option and a status property
			ended: 0

/*		Persistant status properties created dynamically at _init():
			width
			height
			cssClass
			nativeVideoControls
			noFullWindow
			noVolume
			playbackRateEnabled // Warning - Technically, we can have both Flash and HTML, so this might not be correct if the Flash is active. That is a niche case.
*/
		},

		internal: { // Instanced in _init()
			ready: false
			// instance: undefined
			// domNode: undefined
			// htmlDlyCmdId: undefined
			// autohideId: undefined
			// mouse: undefined
			// cmdsIgnored
		},
		solution: { // Static Object: Defines the solutions built in dlePlayerVideo.
			html: true,
			aurora: true,
			flash: true
		},
		// 'MPEG-4 support' : canPlayType('video/mp4; codecs="mp4v.20.8"')
		format: { // Static Object
			mp3: {
				codec: 'audio/mpeg',
				flashCanPlay: true,
				media: 'audio'
			},
			m4a: { // AAC / MP4
				codec: 'audio/mp4; codecs="mp4a.40.2"',
				flashCanPlay: true,
				media: 'audio'
			},
			m3u8a: { // AAC / MP4 / Apple HLS
				codec: 'application/vnd.apple.mpegurl; codecs="mp4a.40.2"',
				flashCanPlay: false,
				media: 'audio'
			},
			m3ua: { // M3U
				codec: 'audio/mpegurl',
				flashCanPlay: false,
				media: 'audio'
			},
			oga: { // OGG
				codec: 'audio/ogg; codecs="vorbis, opus"',
				flashCanPlay: false,
				media: 'audio'
			},
			flac: { // FLAC
				codec: 'audio/x-flac',
				flashCanPlay: false,
				media: 'audio'
			},
			wav: { // PCM
				codec: 'audio/wav; codecs="1"',
				flashCanPlay: false,
				media: 'audio'
			},
			webma: { // WEBM
				codec: 'audio/webm; codecs="vorbis"',
				flashCanPlay: false,
				media: 'audio'
			},
			fla: { // FLV / F4A
				codec: 'audio/x-flv',
				flashCanPlay: true,
				media: 'audio'
			},
			rtmpa: { // RTMP AUDIO
				codec: 'audio/rtmp; codecs="rtmp"',
				flashCanPlay: true,
				media: 'audio'
			},
			m4v: { // H.264 / MP4
				codec: 'video/mp4; codecs="avc1.42E01E, mp4a.40.2"',
				flashCanPlay: true,
				media: 'video'
			},
			m3u8v: { // H.264 / AAC / MP4 / Apple HLS
				codec: 'application/vnd.apple.mpegurl; codecs="avc1.42E01E, mp4a.40.2"',
				flashCanPlay: false,
				media: 'video'
			},
			m3uv: { // M3U
				codec: 'audio/mpegurl',
				flashCanPlay: false,
				media: 'video'
			},
			ogv: { // OGG
				codec: 'video/ogg; codecs="theora, vorbis"',
				flashCanPlay: false,
				media: 'video'
			},
			webmv: { // WEBM
				codec: 'video/webm; codecs="vorbis, vp8"',
				flashCanPlay: false,
				media: 'video'
			},
			flv: { // FLV / F4V
				codec: 'video/x-flv',
				flashCanPlay: true,
				media: 'video'
			},
			rtmpv: { // RTMP VIDEO
				codec: 'video/rtmp; codecs="rtmp"',
				flashCanPlay: true,
				media: 'video'
			}
		},
		_init: function() {
			var self = this;

			this.element.empty();

			this.status = $.extend({}, this.status); // Copy static to unique instance.
			this.internal = $.extend({}, this.internal); // Copy static to unique instance.

			// Initialize the time format
			this.options.timeFormat = $.extend({}, $.dlePlayerVideo.timeFormat, this.options.timeFormat);

			// On iOS, assume commands will be ignored before user initiates them.
			this.internal.cmdsIgnored = $.dlePlayerVideo.platform.ipad || $.dlePlayerVideo.platform.iphone || $.dlePlayerVideo.platform.ipod;

			this.internal.domNode = this.element.get(0);

			// Add key bindings focus to 1st dlePlayerVideo instanced with key control enabled.
			if(this.options.keyEnabled && !$.dlePlayerVideo.focus) {
				$.dlePlayerVideo.focus = this;
			}

			// A fix for Android where older (2.3) and even some 4.x devices fail to work when changing the *audio* SRC and then playing immediately.
			this.androidFix = {
				setMedia: false, // True when media set
				play: false, // True when a progress event will instruct the media to play
				pause: false, // True when a progress event will instruct the media to pause at a time.
				time: NaN // The play(time) parameter
			};
			if($.dlePlayerVideo.platform.android) {
				this.options.preload = this.options.preload !== 'auto' ? 'metadata' : 'auto'; // Default to metadata, but allow auto.
			}

			this.formats = []; // Array based on supplied string option. Order defines priority.
			this.solutions = []; // Array based on solution string option. Order defines priority.
			this.require = {}; // Which media types are required: video, audio.

			this.htmlElement = {}; // DOM elements created by dlePlayerVideo
			this.html = {}; // In _init()'s this.desired code and setmedia(): Accessed via this[solution], where solution from this.solutions array.
			this.html.audio = {};
			this.html.video = {};
			this.aurora = {}; // In _init()'s this.desired code and setmedia(): Accessed via this[solution], where solution from this.solutions array.
			this.aurora.formats = [];
			this.aurora.properties = [];
			this.flash = {}; // In _init()'s this.desired code and setmedia(): Accessed via this[solution], where solution from this.solutions array.

			this.css = {};
			this.css.cs = {}; // Holds the css selector strings
			this.css.jq = {}; // Holds jQuery selectors. ie., $(css.cs.method)

			this.ancestorJq = []; // Holds jQuery selector of cssSelectorAncestor. Init would use $() instead of [], but it is only 1.4+

			this.options.volume = this._limitValue(this.options.volume, 0, 1); // Limit volume value's bounds.

			// Create the formats array, with prority based on the order of the supplied formats string
			$.each(this.options.supplied.toLowerCase().split(","), function(index1, value1) {
				var format = value1.replace(/^\s+|\s+$/g, ""); //trim
				if(self.format[format]) { // Check format is valid.
					var dupFound = false;
					$.each(self.formats, function(index2, value2) { // Check for duplicates
						if(format === value2) {
							dupFound = true;
							return false;
						}
					});
					if(!dupFound) {
						self.formats.push(format);
					}
				}
			});

			// Create the solutions array, with prority based on the order of the solution string
			$.each(this.options.solution.toLowerCase().split(","), function(index1, value1) {
				var solution = value1.replace(/^\s+|\s+$/g, ""); //trim
				if(self.solution[solution]) { // Check solution is valid.
					var dupFound = false;
					$.each(self.solutions, function(index2, value2) { // Check for duplicates
						if(solution === value2) {
							dupFound = true;
							return false;
						}
					});
					if(!dupFound) {
						self.solutions.push(solution);
					}
				}
			});

			// Create Aurora.js formats array
			$.each(this.options.auroraFormats.toLowerCase().split(","), function(index1, value1) {
				var format = value1.replace(/^\s+|\s+$/g, ""); //trim
				if(self.format[format]) { // Check format is valid.
					var dupFound = false;
					$.each(self.aurora.formats, function(index2, value2) { // Check for duplicates
						if(format === value2) {
							dupFound = true;
							return false;
						}
					});
					if(!dupFound) {
						self.aurora.formats.push(format);
					}
				}
			});

			this.internal.instance = "jp_video_" + this.count;
			this.instances[this.internal.instance] = this.element;

			// Check the dlePlayerVideo div has an id and create one if required. Important for Flash to know the unique id for comms.
			if(!this.element.attr("id")) {
				this.element.attr("id", this.options.idPrefix + "_jplayer_" + this.count);
			}

			this.internal.self = $.extend({}, {
				id: this.element.attr("id"),
				jq: this.element
			});
			this.internal.audio = $.extend({}, {
				id: this.options.idPrefix + "_audio_" + this.count,
				jq: undefined
			});
			this.internal.video = $.extend({}, {
				id: this.options.idPrefix + "_video_" + this.count,
				jq: undefined
			});
			this.internal.flash = $.extend({}, {
				id: this.options.idPrefix + "_flash_" + this.count,
				jq: undefined,
				swf: this.options.swfPath + (this.options.swfPath.toLowerCase().slice(-4) !== ".swf" ? (this.options.swfPath && this.options.swfPath.slice(-1) !== "/" ? "/" : "") + "jquery.jplayer.swf" : "")
			});
			this.internal.poster = $.extend({}, {
				id: this.options.idPrefix + "_poster_" + this.count,
				jq: undefined
			});

			// Register listeners defined in the constructor
			$.each($.dlePlayerVideo.event, function(eventName,eventType) {
				if(self.options[eventName] !== undefined) {
					self.element.bind(eventType + ".dlePlayerVideo", self.options[eventName]); // With .dlePlayerVideo namespace.
					self.options[eventName] = undefined; // Destroy the handler pointer copy on the options. Reason, events can be added/removed in other ways so this could be obsolete and misleading.
				}
			});

			// Determine if we require solutions for audio, video or both media types.
			this.require.audio = false;
			this.require.video = false;
			$.each(this.formats, function(priority, format) {
				self.require[self.format[format].media] = true;
			});

			// Now required types are known, finish the options default settings.
			if(this.require.video) {
				this.options = $.extend(true, {},
					this.optionsVideo,
					this.options
				);
			} else {
				this.options = $.extend(true, {},
					this.optionsAudio,
					this.options
				);
			}
			this._setSize(); // update status and dlePlayerVideo element size

			// Determine the status for Blocklisted options.
			this.status.nativeVideoControls = this._uaBlocklist(this.options.nativeVideoControls);
			this.status.noFullWindow = this._uaBlocklist(this.options.noFullWindow);
			this.status.noVolume = this._uaBlocklist(this.options.noVolume);

			// Create event handlers if native fullscreen is supported
			if($.dlePlayerVideo.nativeFeatures.fullscreen.api.fullscreenEnabled) {
				this._fullscreenAddEventListeners();
			}

			// The native controls are only for video and are disabled when audio is also used.
			this._restrictNativeVideoControls();

			// Create the poster image.
			this.htmlElement.poster = document.createElement('img');
			this.htmlElement.poster.id = this.internal.poster.id;
			this.htmlElement.poster.onload = function() { // Note that this did not work on Firefox 3.6: poster.addEventListener("onload", function() {}, false); Did not investigate x-browser.
				if(!self.status.video || self.status.waitForPlay) {
					self.internal.poster.jq.show();
				}
			};
			this.element.append(this.htmlElement.poster);
			this.internal.poster.jq = $("#" + this.internal.poster.id);
			this.internal.poster.jq.css({'width': this.status.width, 'height': this.status.height});
			this.internal.poster.jq.hide();
			this.internal.poster.jq.bind("click.dlePlayerVideo", function() {
				self._trigger($.dlePlayerVideo.event.click);
			});

			// Generate the required media elements
			this.html.audio.available = false;
			if(this.require.audio) { // If a supplied format is audio
				this.htmlElement.audio = document.createElement('audio');
				this.htmlElement.audio.id = this.internal.audio.id;
				this.html.audio.available = !!this.htmlElement.audio.canPlayType && this._testCanPlayType(this.htmlElement.audio); // Test is for IE9 on Win Server 2008.
			}
			this.html.video.available = false;
			if(this.require.video) { // If a supplied format is video
				this.htmlElement.video = document.createElement('video');
				this.htmlElement.video.id = this.internal.video.id;
				this.html.video.available = !!this.htmlElement.video.canPlayType && this._testCanPlayType(this.htmlElement.video); // Test is for IE9 on Win Server 2008.
			}

			this.flash.available = this._checkForFlash(10.1);

			this.html.canPlay = {};
			this.aurora.canPlay = {};
			this.flash.canPlay = {};
			$.each(this.formats, function(priority, format) {
				self.html.canPlay[format] = self.html[self.format[format].media].available && "" !== self.htmlElement[self.format[format].media].canPlayType(self.format[format].codec);
				self.aurora.canPlay[format] = ($.inArray(format, self.aurora.formats) > -1);
				self.flash.canPlay[format] = self.format[format].flashCanPlay && self.flash.available;
			});
			this.html.desired = false;
			this.aurora.desired = false;
			this.flash.desired = false;
			$.each(this.solutions, function(solutionPriority, solution) {
				if(solutionPriority === 0) {
					self[solution].desired = true;
				} else {
					var audioCanPlay = false;
					var videoCanPlay = false;
					$.each(self.formats, function(formatPriority, format) {
						if(self[self.solutions[0]].canPlay[format]) { // The other solution can play
							if(self.format[format].media === 'video') {
								videoCanPlay = true;
							} else {
								audioCanPlay = true;
							}
						}
					});
					self[solution].desired = (self.require.audio && !audioCanPlay) || (self.require.video && !videoCanPlay);
				}
			});
			// This is what dlePlayerVideo will support, based on solution and supplied.
			this.html.support = {};
			this.aurora.support = {};
			this.flash.support = {};
			$.each(this.formats, function(priority, format) {
				self.html.support[format] = self.html.canPlay[format] && self.html.desired;
				self.aurora.support[format] = self.aurora.canPlay[format] && self.aurora.desired;
				self.flash.support[format] = self.flash.canPlay[format] && self.flash.desired;
			});
			// If dlePlayerVideo is supporting any format in a solution, then the solution is used.
			this.html.used = false;
			this.aurora.used = false;
			this.flash.used = false;
			$.each(this.solutions, function(solutionPriority, solution) {
				$.each(self.formats, function(formatPriority, format) {
					if(self[solution].support[format]) {
						self[solution].used = true;
						return false;
					}
				});
			});

			// Init solution active state and the event gates to false.
			this._resetActive();
			this._resetGate();

			// Set up the css selectors for the control and feedback entities.
			this._cssSelectorAncestor(this.options.cssSelectorAncestor);

			// If neither html nor aurora nor flash are being used by this browser, then media playback is not possible. Trigger an error event.
			if(!(this.html.used || this.aurora.used || this.flash.used)) {
				this._error( {
					type: $.dlePlayerVideo.error.NO_SOLUTION,
					context: "{solution:'" + this.options.solution + "', supplied:'" + this.options.supplied + "'}",
					message: $.dlePlayerVideo.errorMsg.NO_SOLUTION,
					hint: $.dlePlayerVideo.errorHint.NO_SOLUTION
				});
				if(this.css.jq.noSolution.length) {
					this.css.jq.noSolution.show();
				}
			} else {
				if(this.css.jq.noSolution.length) {
					this.css.jq.noSolution.hide();
				}
			}

			// Add the flash solution if it is being used.
			if(this.flash.used) {
				var htmlObj,
				flashVars = 'jQuery=' + encodeURI(this.options.noConflict) + '&id=' + encodeURI(this.internal.self.id) + '&vol=' + this.options.volume + '&muted=' + this.options.muted;

				// Code influenced by SWFObject 2.2: http://code.google.com/p/swfobject/
				// Non IE browsers have an initial Flash size of 1 by 1 otherwise the wmode affected the Flash ready event.

				if($.dlePlayerVideo.browser.msie && (Number($.dlePlayerVideo.browser.version) < 9 || $.dlePlayerVideo.browser.documentMode < 9)) {
					var objStr = '<object id="' + this.internal.flash.id + '" classid="clsid:d27cdb6e-ae6d-11cf-96b8-444553540000" width="0" height="0" tabindex="-1"></object>';

					var paramStr = [
						'<param name="movie" value="' + this.internal.flash.swf + '" />',
						'<param name="FlashVars" value="' + flashVars + '" />',
						'<param name="allowScriptAccess" value="always" />',
						'<param name="bgcolor" value="' + this.options.backgroundColor + '" />',
						'<param name="wmode" value="' + this.options.wmode + '" />'
					];

					htmlObj = document.createElement(objStr);
					for(var i=0; i < paramStr.length; i++) {
						htmlObj.appendChild(document.createElement(paramStr[i]));
					}
				} else {
					var createParam = function(el, n, v) {
						var p = document.createElement("param");
						p.setAttribute("name", n);
						p.setAttribute("value", v);
						el.appendChild(p);
					};

					htmlObj = document.createElement("object");
					htmlObj.setAttribute("id", this.internal.flash.id);
					htmlObj.setAttribute("name", this.internal.flash.id);
					htmlObj.setAttribute("data", this.internal.flash.swf);
					htmlObj.setAttribute("type", "application/x-shockwave-flash");
					htmlObj.setAttribute("width", "1"); // Non-zero
					htmlObj.setAttribute("height", "1"); // Non-zero
					htmlObj.setAttribute("tabindex", "-1");
					createParam(htmlObj, "flashvars", flashVars);
					createParam(htmlObj, "allowscriptaccess", "always");
					createParam(htmlObj, "bgcolor", this.options.backgroundColor);
					createParam(htmlObj, "wmode", this.options.wmode);
				}

				this.element.append(htmlObj);
				this.internal.flash.jq = $(htmlObj);
			}

			// Setup playbackRate ability before using _addHtmlEventListeners()
			if(this.html.used && !this.flash.used) { // If only HTML
				// Using the audio element capabilities for playbackRate. ie., Assuming video element is the same.
				this.status.playbackRateEnabled = this._testPlaybackRate('audio');
			} else {
				this.status.playbackRateEnabled = false;
			}

			this._updatePlaybackRate();

			// Add the HTML solution if being used.
			if(this.html.used) {

				// The HTML Audio handlers
				if(this.html.audio.available) {
					this._addHtmlEventListeners(this.htmlElement.audio, this.html.audio);
					this.element.append(this.htmlElement.audio);
					this.internal.audio.jq = $("#" + this.internal.audio.id);
				}

				// The HTML Video handlers
				if(this.html.video.available) {
					this._addHtmlEventListeners(this.htmlElement.video, this.html.video);
					this.element.append(this.htmlElement.video);
					this.internal.video.jq = $("#" + this.internal.video.id);
					if(this.status.nativeVideoControls) {
						this.internal.video.jq.css({'width': this.status.width, 'height': this.status.height});
					} else {
						this.internal.video.jq.css({'width':'0px', 'height':'0px'}); // Using size 0x0 since a .hide() causes issues in iOS
					}
					this.internal.video.jq.bind("click.dlePlayerVideo", function() {
						self._trigger($.dlePlayerVideo.event.click);
					});
				}
			}

			// Add the Aurora.js solution if being used.
			if(this.aurora.used) {
				// Aurora.js player need to be created for each media, see setMedia function.
			}

			// Create the bridge that emulates the HTML Media element on the dlePlayerVideo DIV
			if( this.options.emulateHtml ) {
				this._emulateHtmlBridge();
			}

			if((this.html.used || this.aurora.used) && !this.flash.used) { // If only HTML, then emulate flash ready() call after 100ms.
				setTimeout( function() {
					self.internal.ready = true;
					self.version.flash = "n/a";
					self._trigger($.dlePlayerVideo.event.repeat); // Trigger the repeat event so its handler can initialize itself with the loop option.
					self._trigger($.dlePlayerVideo.event.ready);
				}, 100);
			}

			// Initialize the interface components with the options.
			this._updateNativeVideoControls();
			// The other controls are now setup in _cssSelectorAncestor()
			if(this.css.jq.videoPlay.length) {
				this.css.jq.videoPlay.hide();
			}

			$.dlePlayerVideo.prototype.count++; // Change static variable via prototype.
		},
		destroy: function() {
			// MJP: The background change remains. Would need to store the original to restore it correctly.
			// MJP: The dlePlayerVideo element's size change remains.

			// Clear the media to reset the GUI and stop any downloads. Streams on some browsers had persited. (Chrome)
			this.clearMedia();
			// Remove the size/sizeFull cssClass from the cssSelectorAncestor
			this._removeUiClass();
			// Remove the times from the GUI
			if(this.css.jq.currentTime.length) {
				this.css.jq.currentTime.text("");
			}
			if(this.css.jq.duration.length) {
				this.css.jq.duration.text("");
			}
			// Remove any bindings from the interface controls.
			$.each(this.css.jq, function(fn, jq) {
				// Check selector is valid before trying to execute method.
				if(jq.length) {
					jq.unbind(".dlePlayerVideo");
				}
			});
			// Remove the click handlers for $.dlePlayerVideo.event.click
			this.internal.poster.jq.unbind(".dlePlayerVideo");
			if(this.internal.video.jq) {
				this.internal.video.jq.unbind(".dlePlayerVideo");
			}
			// Remove the fullscreen event handlers
			this._fullscreenRemoveEventListeners();
			// Remove key bindings
			if(this === $.dlePlayerVideo.focus) {
				$.dlePlayerVideo.focus = null;
			}
			// Destroy the HTML bridge.
			if(this.options.emulateHtml) {
				this._destroyHtmlBridge();
			}
			this.element.removeData("dlePlayerVideo"); // Remove dlePlayerVideo data
			this.element.unbind(".dlePlayerVideo"); // Remove all event handlers created by the dlePlayerVideo constructor
			this.element.empty(); // Remove the inserted child elements

			delete this.instances[this.internal.instance]; // Clear the instance on the static instance object
		},
		destroyRemoved: function() { // Destroy any instances that have gone away.
			var self = this;
			$.each(this.instances, function(i, element) {
				if(self.element !== element) { // Do not destroy this instance.
					if(!element.data("dlePlayerVideo")) { // Check that element is a real dlePlayerVideo.
						element.dlePlayerVideo("destroy");
						delete self.instances[i];
					}
				}
			});
		},
		enable: function() { // Plan to implement
			// options.disabled = false
		},
		disable: function () { // Plan to implement
			// options.disabled = true
		},
		_testCanPlayType: function(elem) {
			// IE9 on Win Server 2008 did not implement canPlayType(), but it has the property.
			try {
				elem.canPlayType(this.format.mp3.codec); // The type is irrelevant.
				return true;
			} catch(err) {
				return false;
			}
		},
		_testPlaybackRate: function(type) {
			// type: String 'audio' or 'video'
			var el, rate = 0.5;
			type = typeof type === 'string' ? type : 'audio';
			el = document.createElement(type);
			// Wrapping in a try/catch, just in case older HTML5 browsers throw and error.
			try {
				if('playbackRate' in el) {
					el.playbackRate = rate;
					return el.playbackRate === rate;
				} else {
					return false;
				}
			} catch(err) {
				return false;
			}
		},
		_uaBlocklist: function(list) {
			// list : object with properties that are all regular expressions. Property names are irrelevant.
			// Returns true if the user agent is matched in list.
			var	ua = navigator.userAgent.toLowerCase(),
				block = false;

			$.each(list, function(p, re) {
				if(re && re.test(ua)) {
					block = true;
					return false; // exit $.each.
				}
			});
			return block;
		},
		_restrictNativeVideoControls: function() {
			// Fallback to noFullWindow when nativeVideoControls is true and audio media is being used. Affects when both media types are used.
			if(this.require.audio) {
				if(this.status.nativeVideoControls) {
					this.status.nativeVideoControls = false;
					this.status.noFullWindow = true;
				}
			}
		},
		_updateNativeVideoControls: function() {
			if(this.html.video.available && this.html.used) {
				// Turn the HTML Video controls on/off
				this.htmlElement.video.controls = this.status.nativeVideoControls;
				// Show/hide the dlePlayerVideo GUI.
				this._updateAutohide();
				// For when option changed. The poster image is not updated, as it is dealt with in setMedia(). Acceptable degradation since seriously doubt these options will change on the fly. Can again review later.
				if(this.status.nativeVideoControls && this.require.video) {
					this.internal.poster.jq.hide();
					this.internal.video.jq.css({'width': this.status.width, 'height': this.status.height});
				} else if(this.status.waitForPlay && this.status.video) {
					this.internal.poster.jq.show();
					this.internal.video.jq.css({'width': '0px', 'height': '0px'});
				}
			}
		},
		_addHtmlEventListeners: function(mediaElement, entity) {
			var self = this;
			mediaElement.preload = this.options.preload;
			mediaElement.muted = this.options.muted;
			mediaElement.volume = this.options.volume;

			if(this.status.playbackRateEnabled) {
				mediaElement.defaultPlaybackRate = this.options.defaultPlaybackRate;
				mediaElement.playbackRate = this.options.playbackRate;
			}

			// Create the event listeners
			// Only want the active entity to affect dlePlayerVideo and bubble events.
			// Using entity.gate so that object is referenced and gate property always current

			mediaElement.addEventListener("progress", function() {
				if(entity.gate) {
					if(self.internal.cmdsIgnored && this.readyState > 0) { // Detect iOS executed the command
						self.internal.cmdsIgnored = false;
					}
					self._getHtmlStatus(mediaElement);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.progress);
				}
			}, false);
			mediaElement.addEventListener("loadeddata", function() {
				if(entity.gate) {
					self.androidFix.setMedia = false; // Disable the fix after the first progress event.
					if(self.androidFix.play) { // Play Android audio - performing the fix.
						self.androidFix.play = false;
						self.play(self.androidFix.time);
					}
					if(self.androidFix.pause) { // Pause Android audio at time - performing the fix.
						self.androidFix.pause = false;
						self.pause(self.androidFix.time);
					}
					self._trigger($.dlePlayerVideo.event.loadeddata);
				}
			}, false);
			mediaElement.addEventListener("timeupdate", function() {
				if(entity.gate) {
					self._getHtmlStatus(mediaElement);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.timeupdate);
				}
			}, false);
			mediaElement.addEventListener("durationchange", function() {
				if(entity.gate) {
					self._getHtmlStatus(mediaElement);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.durationchange);
				}
			}, false);
			mediaElement.addEventListener("play", function() {
				if(entity.gate) {
					self._updateButtons(true);
					self._html_checkWaitForPlay(); // So the native controls update this variable and puts the hidden interface in the correct state. Affects toggling native controls.
					self._trigger($.dlePlayerVideo.event.play);
				}
			}, false);
			mediaElement.addEventListener("playing", function() {
				if(entity.gate) {
					self._updateButtons(true);
					self._seeked();
					self._trigger($.dlePlayerVideo.event.playing);
				}
			}, false);
			mediaElement.addEventListener("pause", function() {
				if(entity.gate) {
					self._updateButtons(false);
					self._trigger($.dlePlayerVideo.event.pause);
				}
			}, false);
			mediaElement.addEventListener("waiting", function() {
				if(entity.gate) {
					self._seeking();
					self._trigger($.dlePlayerVideo.event.waiting);
				}
			}, false);
			mediaElement.addEventListener("seeking", function() {
				if(entity.gate) {
					self._seeking();
					self._trigger($.dlePlayerVideo.event.seeking);
				}
			}, false);
			mediaElement.addEventListener("seeked", function() {
				if(entity.gate) {
					self._seeked();
					self._trigger($.dlePlayerVideo.event.seeked);
				}
			}, false);
			mediaElement.addEventListener("volumechange", function() {
				if(entity.gate) {
					// Read the values back from the element as the Blackberry PlayBook shares the volume with the physical buttons master volume control.
					// However, when tested 6th July 2011, those buttons do not generate an event. The physical play/pause button does though.
					self.options.volume = mediaElement.volume;
					self.options.muted = mediaElement.muted;
					self._updateMute();
					self._updateVolume();
					self._trigger($.dlePlayerVideo.event.volumechange);
				}
			}, false);
			mediaElement.addEventListener("ratechange", function() {
				if(entity.gate) {
					self.options.defaultPlaybackRate = mediaElement.defaultPlaybackRate;
					self.options.playbackRate = mediaElement.playbackRate;
					self._updatePlaybackRate();
					self._trigger($.dlePlayerVideo.event.ratechange);
				}
			}, false);
			mediaElement.addEventListener("suspend", function() { // Seems to be the only way of capturing that the iOS4 browser did not actually play the media from the page code. ie., It needs a user gesture.
				if(entity.gate) {
					self._seeked();
					self._trigger($.dlePlayerVideo.event.suspend);
				}
			}, false);
			mediaElement.addEventListener("ended", function() {
				if(entity.gate) {
					// Order of the next few commands are important. Change the time and then pause.
					// Solves a bug in Firefox, where issuing pause 1st causes the media to play from the start. ie., The pause is ignored.
					if(!$.dlePlayerVideo.browser.webkit) { // Chrome crashes if you do this in conjunction with a setMedia command in an ended event handler. ie., The playlist demo.
						self.htmlElement.media.currentTime = 0; // Safari does not care about this command. ie., It works with or without this line. (Both Safari and Chrome are Webkit.)
					}
					self.htmlElement.media.pause(); // Pause otherwise a click on the progress bar will play from that point, when it shouldn't, since it stopped playback.
					self._updateButtons(false);
					self._getHtmlStatus(mediaElement, true); // With override true. Otherwise Chrome leaves progress at full.
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.ended);
				}
			}, false);
			mediaElement.addEventListener("error", function() {
				if(entity.gate) {
					self._updateButtons(false);
					self._seeked();
					if(self.status.srcSet) { // Deals with case of clearMedia() causing an error event.
						clearTimeout(self.internal.htmlDlyCmdId); // Clears any delayed commands used in the HTML solution.
						self.status.waitForLoad = true; // Allows the load operation to try again.
						self.status.waitForPlay = true; // Reset since a play was captured.
						if(self.status.video && !self.status.nativeVideoControls) {
							self.internal.video.jq.css({'width':'0px', 'height':'0px'});
						}
						if(self._validString(self.status.media.poster) && !self.status.nativeVideoControls) {
							self.internal.poster.jq.show();
						}
						if(self.css.jq.videoPlay.length) {
							self.css.jq.videoPlay.show();
						}
						self._error( {
							type: $.dlePlayerVideo.error.URL,
							context: self.status.src, // this.src shows absolute urls. Want context to show the url given.
							message: $.dlePlayerVideo.errorMsg.URL,
							hint: $.dlePlayerVideo.errorHint.URL
						});
					}
				}
			}, false);
			// Create all the other event listeners that bubble up to a dlePlayerVideo event from html, without being used by dlePlayerVideo.
			$.each($.dlePlayerVideo.htmlEvent, function(i, eventType) {
				mediaElement.addEventListener(this, function() {
					if(entity.gate) {
						self._trigger($.dlePlayerVideo.event[eventType]);
					}
				}, false);
			});
		},
		_addAuroraEventListeners : function(player, entity) {
			var self = this;
			//player.preload = this.options.preload;
			//player.muted = this.options.muted;
			player.volume = this.options.volume * 100;

			// Create the event listeners
			// Only want the active entity to affect dlePlayerVideo and bubble events.
			// Using entity.gate so that object is referenced and gate property always current

			player.on("progress", function() {
				if(entity.gate) {
					if(self.internal.cmdsIgnored && this.readyState > 0) { // Detect iOS executed the command
						self.internal.cmdsIgnored = false;
					}
					self._getAuroraStatus(player);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.progress);
					// Progress with song duration, we estimate timeupdate need to be triggered too.
					if (player.duration > 0) {
						self._trigger($.dlePlayerVideo.event.timeupdate);
					}
				}
			}, false);
			player.on("ready", function() {
				if(entity.gate) {
					self._trigger($.dlePlayerVideo.event.loadeddata);
				}
			}, false);
			player.on("duration", function() {
				if(entity.gate) {
					self._getAuroraStatus(player);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.durationchange);
				}
			}, false);
			player.on("end", function() {
				if(entity.gate) {
					// Order of the next few commands are important. Change the time and then pause.
					self._updateButtons(false);
					self._getAuroraStatus(player, true);
					self._updateInterface();
					self._trigger($.dlePlayerVideo.event.ended);
				}
			}, false);
			player.on("error", function() {
				if(entity.gate) {
					self._updateButtons(false);
					self._seeked();
					if(self.status.srcSet) { // Deals with case of clearMedia() causing an error event.
						self.status.waitForLoad = true; // Allows the load operation to try again.
						self.status.waitForPlay = true; // Reset since a play was captured.
						if(self.status.video && !self.status.nativeVideoControls) {
							self.internal.video.jq.css({'width':'0px', 'height':'0px'});
						}
						if(self._validString(self.status.media.poster) && !self.status.nativeVideoControls) {
							self.internal.poster.jq.show();
						}
						if(self.css.jq.videoPlay.length) {
							self.css.jq.videoPlay.show();
						}
						self._error( {
							type: $.dlePlayerVideo.error.URL,
							context: self.status.src, // this.src shows absolute urls. Want context to show the url given.
							message: $.dlePlayerVideo.errorMsg.URL,
							hint: $.dlePlayerVideo.errorHint.URL
						});
					}
				}
			}, false);
		},
		_getHtmlStatus: function(media, override) {
			var ct = 0, cpa = 0, sp = 0, cpr = 0;

			// Fixes the duration bug in iOS, where the durationchange event occurs when media.duration is not always correct.
			// Fixes the initial duration bug in BB OS7, where the media.duration is infinity and displays as NaN:NaN due to Date() using inifity.
			if(isFinite(media.duration)) {
				this.status.duration = media.duration;
			}

			ct = media.currentTime;
			cpa = (this.status.duration > 0) ? 100 * ct / this.status.duration : 0;
			if((typeof media.seekable === "object") && (media.seekable.length > 0)) {
				sp = (this.status.duration > 0) ? 100 * media.seekable.end(media.seekable.length-1) / this.status.duration : 100;
				cpr = (this.status.duration > 0) ? 100 * media.currentTime / media.seekable.end(media.seekable.length-1) : 0; // Duration conditional for iOS duration bug. ie., seekable.end is a NaN in that case.
			} else {
				sp = 100;
				cpr = cpa;
			}

			if(override) {
				ct = 0;
				cpr = 0;
				cpa = 0;
			}

			this.status.seekPercent = sp;
			this.status.currentPercentRelative = cpr;
			this.status.currentPercentAbsolute = cpa;
			this.status.currentTime = ct;

			this.status.remaining = this.status.duration - this.status.currentTime;

			this.status.videoWidth = media.videoWidth;
			this.status.videoHeight = media.videoHeight;

			this.status.readyState = media.readyState;
			this.status.networkState = media.networkState;
			this.status.playbackRate = media.playbackRate;
			this.status.ended = media.ended;
		},
		_getAuroraStatus: function(player, override) {
			var ct = 0, cpa = 0, sp = 0, cpr = 0;

			this.status.duration = player.duration / 1000;

			ct = player.currentTime / 1000;
			cpa = (this.status.duration > 0) ? 100 * ct / this.status.duration : 0;
			if(player.buffered > 0) {
				sp = (this.status.duration > 0) ? (player.buffered * this.status.duration) / this.status.duration : 100;
				cpr = (this.status.duration > 0) ? ct / (player.buffered * this.status.duration) : 0;
			} else {
				sp = 100;
				cpr = cpa;
			}

			if(override) {
				ct = 0;
				cpr = 0;
				cpa = 0;
			}

			this.status.seekPercent = sp;
			this.status.currentPercentRelative = cpr;
			this.status.currentPercentAbsolute = cpa;
			this.status.currentTime = ct;

			this.status.remaining = this.status.duration - this.status.currentTime;

			this.status.readyState = 4; // status.readyState;
			this.status.networkState = 0; // status.networkState;
			this.status.playbackRate = 1; // status.playbackRate;
			this.status.ended = false; // status.ended;
		},
		_resetStatus: function() {
			this.status = $.extend({}, this.status, $.dlePlayerVideo.prototype.status); // Maintains the status properties that persist through a reset.
		},
		_trigger: function(eventType, error, warning) { // eventType always valid as called using $.dlePlayerVideo.event.eventType
			var event = $.Event(eventType);
			event.dlePlayerVideo = {};
			event.dlePlayerVideo.version = $.extend({}, this.version);
			event.dlePlayerVideo.options = $.extend(true, {}, this.options); // Deep copy
			event.dlePlayerVideo.status = $.extend(true, {}, this.status); // Deep copy
			event.dlePlayerVideo.html = $.extend(true, {}, this.html); // Deep copy
			event.dlePlayerVideo.aurora = $.extend(true, {}, this.aurora); // Deep copy
			event.dlePlayerVideo.flash = $.extend(true, {}, this.flash); // Deep copy
			if(error) {
				event.dlePlayerVideo.error = $.extend({}, error);
			}
			if(warning) {
				event.dlePlayerVideo.warning = $.extend({}, warning);
			}
			this.element.trigger(event);
		},
		dlePlayerVideoFlashEvent: function(eventType, status) { // Called from Flash
			if(eventType === $.dlePlayerVideo.event.ready) {
				if(!this.internal.ready) {
					this.internal.ready = true;
					this.internal.flash.jq.css({'width':'0px', 'height':'0px'}); // Once Flash generates the ready event, minimise to zero as it is not affected by wmode anymore.

					this.version.flash = status.version;
					if(this.version.needFlash !== this.version.flash) {
						this._error( {
							type: $.dlePlayerVideo.error.VERSION,
							context: this.version.flash,
							message: $.dlePlayerVideo.errorMsg.VERSION + this.version.flash,
							hint: $.dlePlayerVideo.errorHint.VERSION
						});
					}
					this._trigger($.dlePlayerVideo.event.repeat); // Trigger the repeat event so its handler can initialize itself with the loop option.
					this._trigger(eventType);
				} else {
					// This condition occurs if the Flash is hidden and then shown again.
					// Firefox also reloads the Flash if the CSS position changes. position:fixed is used for full screen.

					// Only do this if the Flash is the solution being used at the moment. Affects Media players where both solution may be being used.
					if(this.flash.gate) {

						// Send the current status to the Flash now that it is ready (available) again.
						if(this.status.srcSet) {

							// Need to read original status before issuing the setMedia command.
							var	currentTime = this.status.currentTime,
								paused = this.status.paused;

							this.setMedia(this.status.media);
							this.volumeWorker(this.options.volume);
							if(currentTime > 0) {
								if(paused) {
									this.pause(currentTime);
								} else {
									this.play(currentTime);
								}
							}
						}
						this._trigger($.dlePlayerVideo.event.flashreset);
					}
				}
			}
			if(this.flash.gate) {
				switch(eventType) {
					case $.dlePlayerVideo.event.progress:
						this._getFlashStatus(status);
						this._updateInterface();
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.timeupdate:
						this._getFlashStatus(status);
						this._updateInterface();
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.play:
						this._seeked();
						this._updateButtons(true);
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.pause:
						this._updateButtons(false);
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.ended:
						this._updateButtons(false);
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.click:
						this._trigger(eventType); // This could be dealt with by the default
						break;
					case $.dlePlayerVideo.event.error:
						this.status.waitForLoad = true; // Allows the load operation to try again.
						this.status.waitForPlay = true; // Reset since a play was captured.
						if(this.status.video) {
							this.internal.flash.jq.css({'width':'0px', 'height':'0px'});
						}
						if(this._validString(this.status.media.poster)) {
							this.internal.poster.jq.show();
						}
						if(this.css.jq.videoPlay.length && this.status.video) {
							this.css.jq.videoPlay.show();
						}
						if(this.status.video) { // Set up for another try. Execute before error event.
							this._flash_setVideo(this.status.media);
						} else {
							this._flash_setAudio(this.status.media);
						}
						this._updateButtons(false);
						this._error( {
							type: $.dlePlayerVideo.error.URL,
							context:status.src,
							message: $.dlePlayerVideo.errorMsg.URL,
							hint: $.dlePlayerVideo.errorHint.URL
						});
						break;
					case $.dlePlayerVideo.event.seeking:
						this._seeking();
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.seeked:
						this._seeked();
						this._trigger(eventType);
						break;
					case $.dlePlayerVideo.event.ready:
						// The ready event is handled outside the switch statement.
						// Captured here otherwise 2 ready events would be generated if the ready event handler used setMedia.
						break;
					default:
						this._trigger(eventType);
				}
			}
			return false;
		},
		_getFlashStatus: function(status) {
			this.status.seekPercent = status.seekPercent;
			this.status.currentPercentRelative = status.currentPercentRelative;
			this.status.currentPercentAbsolute = status.currentPercentAbsolute;
			this.status.currentTime = status.currentTime;
			this.status.duration = status.duration;
			this.status.remaining = status.duration - status.currentTime;

			this.status.videoWidth = status.videoWidth;
			this.status.videoHeight = status.videoHeight;

			// The Flash does not generate this information in this release
			this.status.readyState = 4; // status.readyState;
			this.status.networkState = 0; // status.networkState;
			this.status.playbackRate = 1; // status.playbackRate;
			this.status.ended = false; // status.ended;
		},
		_updateButtons: function(playing) {
			if(playing === undefined) {
				playing = !this.status.paused;
			} else {
				this.status.paused = !playing;
			}
			// Apply the state classes. (For the useStateClassSkin:true option)
			if(playing) {
				this.addStateClass('playing');
			} else {
				this.removeStateClass('playing');
			}
			if(!this.status.noFullWindow && this.options.fullWindow) {
				this.addStateClass('fullScreen');
			} else {
				this.removeStateClass('fullScreen');
			}
			if(this.options.loop) {
				this.addStateClass('looped');
			} else {
				this.removeStateClass('looped');
			}
			// Toggle the GUI element pairs. (For the useStateClassSkin:false option)
			if(this.css.jq.play.length && this.css.jq.pause.length) {
				if(playing) {
					this.css.jq.play.hide();
					this.css.jq.pause.show();
				} else {
					this.css.jq.play.show();
					this.css.jq.pause.hide();
				}
			}
			if(this.css.jq.restoreScreen.length && this.css.jq.fullScreen.length) {
				if(this.status.noFullWindow) {
					this.css.jq.fullScreen.hide();
					this.css.jq.restoreScreen.hide();
				} else if(this.options.fullWindow) {
					this.css.jq.fullScreen.hide();
					this.css.jq.restoreScreen.show();
				} else {
					this.css.jq.fullScreen.show();
					this.css.jq.restoreScreen.hide();
				}
			}
			if(this.css.jq.repeat.length && this.css.jq.repeatOff.length) {
				if(this.options.loop) {
					this.css.jq.repeat.hide();
					this.css.jq.repeatOff.show();
				} else {
					this.css.jq.repeat.show();
					this.css.jq.repeatOff.hide();
				}
			}
		},
		_updateInterface: function() {
			if(this.css.jq.seekBar.length) {
				this.css.jq.seekBar.width(this.status.seekPercent+"%");
			}
			if(this.css.jq.playBar.length) {
				if(this.options.smoothPlayBar) {
					this.css.jq.playBar.stop().animate({
						width: this.status.currentPercentAbsolute+"%"
					}, 250, "linear");
				} else {
					this.css.jq.playBar.width(this.status.currentPercentRelative+"%");
				}
			}
			var currentTimeText = '';
			if(this.css.jq.currentTime.length) {
				currentTimeText = this._convertTime(this.status.currentTime);
				if(currentTimeText !== this.css.jq.currentTime.text()) {
					this.css.jq.currentTime.text(this._convertTime(this.status.currentTime));
				}
			}
			var durationText = '',
				duration = this.status.duration,
				remaining = this.status.remaining;
			if(this.css.jq.duration.length) {
				if(typeof this.status.media.duration === 'string') {
					durationText = this.status.media.duration;
				} else {
					if(typeof this.status.media.duration === 'number') {
						duration = this.status.media.duration;
						remaining = duration - this.status.currentTime;
					}
					if(this.options.remainingDuration) {
						durationText = (remaining > 0 ? '-' : '') + this._convertTime(remaining);
					} else {
						durationText = this._convertTime(duration);
					}
				}
				if(durationText !== this.css.jq.duration.text()) {
					this.css.jq.duration.text(durationText);
				}
			}
		},
		_convertTime: ConvertTime.prototype.time,
		_seeking: function() {
			if(this.css.jq.seekBar.length) {
				this.css.jq.seekBar.addClass("mdtc-clnplrv-seeking-bg");
			}
			this.addStateClass('seeking');
		},
		_seeked: function() {
			if(this.css.jq.seekBar.length) {
				this.css.jq.seekBar.removeClass("mdtc-clnplrv-seeking-bg");
			}
			this.removeStateClass('seeking');
		},
		_resetGate: function() {
			this.html.audio.gate = false;
			this.html.video.gate = false;
			this.aurora.gate = false;
			this.flash.gate = false;
		},
		_resetActive: function() {
			this.html.active = false;
			this.aurora.active = false;
			this.flash.active = false;
		},
		_escapeHtml: function(s) {
			return s.split('&').join('&amp;').split('<').join('&lt;').split('>').join('&gt;').split('"').join('&quot;');
		},
		_qualifyURL: function(url) {
			var el = document.createElement('div');
			el.innerHTML= '<a href="' + this._escapeHtml(url) + '">x</a>';
			return el.firstChild.href;
		},
		_absoluteMediaUrls: function(media) {
			var self = this;
			$.each(media, function(type, url) {
				if(url && self.format[type] && url.substr(0, 5) !== "data:") {
					media[type] = self._qualifyURL(url);
				}
			});
			return media;
		},
		addStateClass: function(state) {
			if(this.ancestorJq.length) {
				this.ancestorJq.addClass(this.options.stateClass[state]);
			}
		},
		removeStateClass: function(state) {
			if(this.ancestorJq.length) {
				this.ancestorJq.removeClass(this.options.stateClass[state]);
			}
		},
		setMedia: function(media) {

			/*	media[format] = String: URL of format. Must contain all of the supplied option's video or audio formats.
			 *	media.poster = String: Video poster URL.
			 *	media.track = Array: Of objects defining the track element: kind, src, srclang, label, def.
			 *	media.stream = Boolean: * NOT IMPLEMENTED * Designating actual media streams. ie., "false/undefined" for files. Plan to refresh the flash every so often.
			 */

			var	self = this,
				supported = false,
				posterChanged = this.status.media.poster !== media.poster; // Compare before reset. Important for OSX Safari as this.htmlElement.poster.src is absolute, even if original poster URL was relative.

			this._resetMedia();
			this._resetGate();
			this._resetActive();

			// Clear the Android Fix.
			this.androidFix.setMedia = false;
			this.androidFix.play = false;
			this.androidFix.pause = false;

			// Convert all media URLs to absolute URLs.
			media = this._absoluteMediaUrls(media);

			$.each(this.formats, function(formatPriority, format) {
				var isVideo = self.format[format].media === 'video';
				$.each(self.solutions, function(solutionPriority, solution) {
					if(self[solution].support[format] && self._validString(media[format])) { // Format supported in solution and url given for format.
						var isHtml = solution === 'html';
						var isAurora = solution === 'aurora';

						if(isVideo) {
							if(isHtml) {
								self.html.video.gate = true;
								self._html_setVideo(media);
								self.html.active = true;
							} else {
								self.flash.gate = true;
								self._flash_setVideo(media);
								self.flash.active = true;
							}
							if(self.css.jq.videoPlay.length) {
								self.css.jq.videoPlay.show();
							}
							self.status.video = true;
						} else {
							if(isHtml) {
								self.html.audio.gate = true;
								self._html_setAudio(media);
								self.html.active = true;

								// Setup the Android Fix - Only for HTML audio.
								if($.dlePlayerVideo.platform.android) {
									self.androidFix.setMedia = true;
								}
							} else if(isAurora) {
								self.aurora.gate = true;
								self._aurora_setAudio(media);
								self.aurora.active = true;
							} else {
								self.flash.gate = true;
								self._flash_setAudio(media);
								self.flash.active = true;
							}
							if(self.css.jq.videoPlay.length) {
								self.css.jq.videoPlay.hide();
							}
							self.status.video = false;
						}

						supported = true;
						return false; // Exit $.each
					}
				});
				if(supported) {
					return false; // Exit $.each
				}
			});

			if(supported) {
				if(!(this.status.nativeVideoControls && this.html.video.gate)) {
					// Set poster IMG if native video controls are not being used
					// Note: With IE the IMG onload event occurs immediately when cached.
					// Note: Poster hidden by default in _resetMedia()
					if(this._validString(media.poster)) {
						if(posterChanged) { // Since some browsers do not generate img onload event.
							this.htmlElement.poster.src = media.poster;
						} else {
							this.internal.poster.jq.show();
						}
					}
				}
				if(typeof media.title === 'string') {
					if(this.css.jq.title.length) {
						this.css.jq.title.html(media.title);
					}
				}
				this.status.srcSet = true;
				this.status.media = $.extend({}, media);
				this._updateButtons(false);
				this._updateInterface();
				this._trigger($.dlePlayerVideo.event.setmedia);
			} else { // dlePlayerVideo cannot support any formats provided in this browser
				// Send an error event
				this._error( {
					type: $.dlePlayerVideo.error.NO_SUPPORT,
					context: "{supplied:'" + this.options.supplied + "'}",
					message: $.dlePlayerVideo.errorMsg.NO_SUPPORT,
					hint: $.dlePlayerVideo.errorHint.NO_SUPPORT
				});
			}
		},
		_resetMedia: function() {
			this._resetStatus();
			this._updateButtons(false);
			this._updateInterface();
			this._seeked();
			this.internal.poster.jq.hide();

			clearTimeout(this.internal.htmlDlyCmdId);

			if(this.html.active) {
				this._html_resetMedia();
			} else if(this.aurora.active) {
				this._aurora_resetMedia();
			} else if(this.flash.active) {
				this._flash_resetMedia();
			}
		},
		clearMedia: function() {
			this._resetMedia();

			if(this.html.active) {
				this._html_clearMedia();
			} else if(this.aurora.active) {
				this._aurora_clearMedia();
			} else if(this.flash.active) {
				this._flash_clearMedia();
			}

			this._resetGate();
			this._resetActive();
		},
		load: function() {
			if(this.status.srcSet) {
				if(this.html.active) {
					this._html_load();
				} else if(this.aurora.active) {
					this._aurora_load();
				} else if(this.flash.active) {
					this._flash_load();
				}
			} else {
				this._urlNotSetError("load");
			}
		},
		focus: function() {
			if(this.options.keyEnabled) {
				$.dlePlayerVideo.focus = this;
			}
		},
		play: function(time) {
			var guiAction = typeof time === "object"; // Flags GUI click events so we know this was not a direct command, but an action taken by the user on the GUI.
			if(guiAction && this.options.useStateClassSkin && !this.status.paused) {
				this.pause(time); // The time would be the click event, but passing it over so info is not lost.
			} else {
				time = (typeof time === "number") ? time : NaN; // Remove jQuery event from click handler
				if(this.status.srcSet) {
					this.focus();
					if(this.html.active) {
						this._html_play(time);
					} else if(this.aurora.active) {
						this._aurora_play(time);
					} else if(this.flash.active) {
						this._flash_play(time);
					}
				} else {
					this._urlNotSetError("play");
				}
			}
		},
		videoPlay: function() { // Handles clicks on the play button over the video poster
			this.play();
		},
		pause: function(time) {
			time = (typeof time === "number") ? time : NaN; // Remove jQuery event from click handler
			if(this.status.srcSet) {
				if(this.html.active) {
					this._html_pause(time);
				} else if(this.aurora.active) {
					this._aurora_pause(time);
				} else if(this.flash.active) {
					this._flash_pause(time);
				}
			} else {
				this._urlNotSetError("pause");
			}
		},
		tellOthers: function(command, conditions) {
			var self = this,
				hasConditions = typeof conditions === 'function',
				args = Array.prototype.slice.call(arguments); // Convert arguments to an Array.

			if(typeof command !== 'string') { // Ignore, since no command.
				return; // Return undefined to maintain chaining.
			}
			if(hasConditions) {
				args.splice(1, 1); // Remove the conditions from the arguments
			}

			$.dlePlayerVideo.prototype.destroyRemoved();
			$.each(this.instances, function() {
				// Remember that "this" is the instance's "element" in the $.each() loop.
				if(self.element !== this) { // Do not tell my instance.
					if(!hasConditions || conditions.call(this.data("dlePlayerVideo"), self)) {
						this.dlePlayerVideo.apply(this, args);
					}
				}
			});
		},
		pauseOthers: function(time) {
			this.tellOthers("pause", function() {
				// In the conditions function, the "this" context is the other instance's dlePlayerVideo object.
				return this.status.srcSet;
			}, time);
		},
		stop: function() {
			if(this.status.srcSet) {
				if(this.html.active) {
					this._html_pause(0);
				} else if(this.aurora.active) {
					this._aurora_pause(0);
				} else if(this.flash.active) {
					this._flash_pause(0);
				}
			} else {
				this._urlNotSetError("stop");
			}
		},
		playHead: function(p) {
			p = this._limitValue(p, 0, 100);
			if(this.status.srcSet) {
				if(this.html.active) {
					this._html_playHead(p);
				} else if(this.aurora.active) {
					this._aurora_playHead(p);
				} else if(this.flash.active) {
					this._flash_playHead(p);
				}
			} else {
				this._urlNotSetError("playHead");
			}
		},
		_muted: function(muted) {
			this.mutedWorker(muted);
			if(this.options.globalVolume) {
				this.tellOthers("mutedWorker", function() {
					// Check the other instance has global volume enabled.
					return this.options.globalVolume;
				}, muted);
			}
		},
		mutedWorker: function(muted) {
			this.options.muted = muted;
			if(this.html.used) {
				this._html_setProperty('muted', muted);
			}
			if(this.aurora.used) {
				this._aurora_mute(muted);
			}
			if(this.flash.used) {
				this._flash_mute(muted);
			}

			// The HTML solution generates this event from the media element itself.
			if(!this.html.video.gate && !this.html.audio.gate) {
				this._updateMute(muted);
				this._updateVolume(this.options.volume);
				this._trigger($.dlePlayerVideo.event.volumechange);
			}
		},
		mute: function(mute) { // mute is either: undefined (true), an event object (true) or a boolean (muted).
			var guiAction = typeof mute === "object"; // Flags GUI click events so we know this was not a direct command, but an action taken by the user on the GUI.
			if(guiAction && this.options.useStateClassSkin && this.options.muted) {
				this._muted(false);
			} else {
				mute = mute === undefined ? true : !!mute;
				this._muted(mute);
			}
		},
		unmute: function(unmute) { // unmute is either: undefined (true), an event object (true) or a boolean (!muted).
			unmute = unmute === undefined ? true : !!unmute;
			this._muted(!unmute);
		},
		_updateMute: function(mute) {
			if(mute === undefined) {
				mute = this.options.muted;
			}
			if(mute) {
				this.addStateClass('muted');
			} else {
				this.removeStateClass('muted');
			}
			if(this.css.jq.mute.length && this.css.jq.unmute.length) {
				if(this.status.noVolume) {
					this.css.jq.mute.hide();
					this.css.jq.unmute.hide();
				} else if(mute) {
					this.css.jq.mute.hide();
					this.css.jq.unmute.show();
				} else {
					this.css.jq.mute.show();
					this.css.jq.unmute.hide();
				}
			}
		},
		volume: function(v) {
			this.volumeWorker(v);
			if(this.options.globalVolume) {
				this.tellOthers("volumeWorker", function() {
					// Check the other instance has global volume enabled.
					return this.options.globalVolume;
				}, v);
			}
		},
		volumeWorker: function(v) {
			v = this._limitValue(v, 0, 1);
			this.options.volume = v;

			if(this.html.used) {
				this._html_setProperty('volume', v);
			}
			if(this.aurora.used) {
				this._aurora_volume(v);
			}
			if(this.flash.used) {
				this._flash_volume(v);
			}

			// The HTML solution generates this event from the media element itself.
			if(!this.html.video.gate && !this.html.audio.gate) {
				this._updateVolume(v);
				this._trigger($.dlePlayerVideo.event.volumechange);
			}
		},
		volumeBar: function(e) { // Handles clicks on the volumeBar
			if(this.css.jq.volumeBar.length) {
				// Using $(e.currentTarget) to enable multiple volume bars
				var $bar = $(e.currentTarget),
					offset = $bar.offset(),
					x = e.pageX - offset.left,
					w = $bar.width(),
					y = $bar.height() - e.pageY + offset.top,
					h = $bar.height();
				if(this.options.verticalVolume) {
					this.volume(y/h);
				} else {
					this.volume(x/w);
				}
			}
			if(this.options.muted) {
				this._muted(false);
			}
		},
		_updateVolume: function(v) {
			if(v === undefined) {
				v = this.options.volume;
			}
			v = this.options.muted ? 0 : v;

			if(this.status.noVolume) {
				this.addStateClass('noVolume');
				if(this.css.jq.volumeBar.length) {
					this.css.jq.volumeBar.hide();
				}
				if(this.css.jq.volumeBarValue.length) {
					this.css.jq.volumeBarValue.hide();
				}
				if(this.css.jq.volumeMax.length) {
					this.css.jq.volumeMax.hide();
				}
			} else {
				this.removeStateClass('noVolume');
				if(this.css.jq.volumeBar.length) {
					this.css.jq.volumeBar.show();
				}
				if(this.css.jq.volumeBarValue.length) {
					this.css.jq.volumeBarValue.show();
					this.css.jq.volumeBarValue[this.options.verticalVolume ? "height" : "width"]((v*100)+"%");
				}
				if(this.css.jq.volumeMax.length) {
					this.css.jq.volumeMax.show();
				}
			}
		},
		volumeMax: function() { // Handles clicks on the volume max
			this.volume(1);
			if(this.options.muted) {
				this._muted(false);
			}
		},
		_cssSelectorAncestor: function(ancestor) {
			var self = this;
			this.options.cssSelectorAncestor = ancestor;
			this._removeUiClass();
			this.ancestorJq = ancestor ? $(ancestor) : []; // Would use $() instead of [], but it is only 1.4+
			if(ancestor && this.ancestorJq.length !== 1) { // So empty strings do not generate the warning.
				this._warning( {
					type: $.dlePlayerVideo.warning.CSS_SELECTOR_COUNT,
					context: ancestor,
					message: $.dlePlayerVideo.warningMsg.CSS_SELECTOR_COUNT + this.ancestorJq.length + " found for cssSelectorAncestor.",
					hint: $.dlePlayerVideo.warningHint.CSS_SELECTOR_COUNT
				});
			}
			this._addUiClass();
			$.each(this.options.cssSelector, function(fn, cssSel) {
				self._cssSelector(fn, cssSel);
			});

			// Set the GUI to the current state.
			this._updateInterface();
			this._updateButtons();
			this._updateAutohide();
			this._updateVolume();
			this._updateMute();
		},
		_cssSelector: function(fn, cssSel) {
			var self = this;
			if(typeof cssSel === 'string') {
				if($.dlePlayerVideo.prototype.options.cssSelector[fn]) {
					if(this.css.jq[fn] && this.css.jq[fn].length) {
						this.css.jq[fn].unbind(".dlePlayerVideo");
					}
					this.options.cssSelector[fn] = cssSel;
					this.css.cs[fn] = this.options.cssSelectorAncestor + " " + cssSel;

					if(cssSel) { // Checks for empty string
						this.css.jq[fn] = $(this.css.cs[fn]);
					} else {
						this.css.jq[fn] = []; // To comply with the css.jq[fn].length check before its use. As of jQuery 1.4 could have used $() for an empty set.
					}

					if(this.css.jq[fn].length && this[fn]) {
						var handler = function(e) {
							e.preventDefault();
							self[fn](e);
							if(self.options.autoBlur) {
								$(this).blur();
							} else {
								$(this).focus(); // Force focus for ARIA.
							}
						};
						this.css.jq[fn].bind("click.dlePlayerVideo", handler); // Using dlePlayerVideo namespace
					}

					if(cssSel && this.css.jq[fn].length !== 1) { // So empty strings do not generate the warning. ie., they just remove the old one.
						this._warning( {
							type: $.dlePlayerVideo.warning.CSS_SELECTOR_COUNT,
							context: this.css.cs[fn],
							message: $.dlePlayerVideo.warningMsg.CSS_SELECTOR_COUNT + this.css.jq[fn].length + " found for " + fn + " method.",
							hint: $.dlePlayerVideo.warningHint.CSS_SELECTOR_COUNT
						});
					}
				} else {
					this._warning( {
						type: $.dlePlayerVideo.warning.CSS_SELECTOR_METHOD,
						context: fn,
						message: $.dlePlayerVideo.warningMsg.CSS_SELECTOR_METHOD,
						hint: $.dlePlayerVideo.warningHint.CSS_SELECTOR_METHOD
					});
				}
			} else {
				this._warning( {
					type: $.dlePlayerVideo.warning.CSS_SELECTOR_STRING,
					context: cssSel,
					message: $.dlePlayerVideo.warningMsg.CSS_SELECTOR_STRING,
					hint: $.dlePlayerVideo.warningHint.CSS_SELECTOR_STRING
				});
			}
		},
		duration: function(e) {
			if(this.options.toggleDuration) {
				if(this.options.captureDuration) {
					e.stopPropagation();
				}
				this._setOption("remainingDuration", !this.options.remainingDuration);
			}
		},
		seekBar: function(e) { // Handles clicks on the seekBar
			if(this.css.jq.seekBar.length) {
				// Using $(e.currentTarget) to enable multiple seek bars
				var $bar = $(e.currentTarget),
					offset = $bar.offset(),
					x = e.pageX - offset.left,
					w = $bar.width(),
					p = 100 * x / w;
				this.playHead(p);
			}
		},
		playbackRate: function(pbr) {
			this._setOption("playbackRate", pbr);
		},
		playbackRateBar: function(e) { // Handles clicks on the playbackRateBar
			if(this.css.jq.playbackRateBar.length) {
				// Using $(e.currentTarget) to enable multiple playbackRate bars
				var $bar = $(e.currentTarget),
					offset = $bar.offset(),
					x = e.pageX - offset.left,
					w = $bar.width(),
					y = $bar.height() - e.pageY + offset.top,
					h = $bar.height(),
					ratio, pbr;
				if(this.options.verticalPlaybackRate) {
					ratio = y/h;
				} else {
					ratio = x/w;
				}
				pbr = ratio * (this.options.maxPlaybackRate - this.options.minPlaybackRate) + this.options.minPlaybackRate;
				this.playbackRate(pbr);
			}
		},
		_updatePlaybackRate: function() {
			var pbr = this.options.playbackRate,
				ratio = (pbr - this.options.minPlaybackRate) / (this.options.maxPlaybackRate - this.options.minPlaybackRate);
			if(this.status.playbackRateEnabled) {
				if(this.css.jq.playbackRateBar.length) {
					this.css.jq.playbackRateBar.show();
				}
				if(this.css.jq.playbackRateBarValue.length) {
					this.css.jq.playbackRateBarValue.show();
					this.css.jq.playbackRateBarValue[this.options.verticalPlaybackRate ? "height" : "width"]((ratio*100)+"%");
				}
			} else {
				if(this.css.jq.playbackRateBar.length) {
					this.css.jq.playbackRateBar.hide();
				}
				if(this.css.jq.playbackRateBarValue.length) {
					this.css.jq.playbackRateBarValue.hide();
				}
			}
		},
		repeat: function(event) { // Handle clicks on the repeat button
			var guiAction = typeof event === "object"; // Flags GUI click events so we know this was not a direct command, but an action taken by the user on the GUI.
			if(guiAction && this.options.useStateClassSkin && this.options.loop) {
				this._loop(false);
			} else {
				this._loop(true);
			}
		},
		repeatOff: function() { // Handle clicks on the repeatOff button
			this._loop(false);
		},
		_loop: function(loop) {
			if(this.options.loop !== loop) {
				this.options.loop = loop;
				this._updateButtons();
				this._trigger($.dlePlayerVideo.event.repeat);
			}
		},

		// Options code adapted from ui.widget.js (1.8.7).  Made changes so the key can use dot notation. To match previous getData solution in dlePlayerVideo 1.
		option: function(key, value) {
			var options = key;

			 // Enables use: options().  Returns a copy of options object
			if ( arguments.length === 0 ) {
				return $.extend( true, {}, this.options );
			}

			if(typeof key === "string") {
				var keys = key.split(".");

				 // Enables use: options("someOption")  Returns a copy of the option. Supports dot notation.
				if(value === undefined) {

					var opt = $.extend(true, {}, this.options);
					for(var i = 0; i < keys.length; i++) {
						if(opt[keys[i]] !== undefined) {
							opt = opt[keys[i]];
						} else {
							this._warning( {
								type: $.dlePlayerVideo.warning.OPTION_KEY,
								context: key,
								message: $.dlePlayerVideo.warningMsg.OPTION_KEY,
								hint: $.dlePlayerVideo.warningHint.OPTION_KEY
							});
							return undefined;
						}
					}
					return opt;
				}

				 // Enables use: options("someOptionObject", someObject}).  Creates: {someOptionObject:someObject}
				 // Enables use: options("someOption", someValue).  Creates: {someOption:someValue}
				 // Enables use: options("someOptionObject.someOption", someValue).  Creates: {someOptionObject:{someOption:someValue}}

				options = {};
				var opts = options;

				for(var j = 0; j < keys.length; j++) {
					if(j < keys.length - 1) {
						opts[keys[j]] = {};
						opts = opts[keys[j]];
					} else {
						opts[keys[j]] = value;
					}
				}
			}

			 // Otherwise enables use: options(optionObject).  Uses original object (the key)

			this._setOptions(options);

			return this;
		},
		_setOptions: function(options) {
			var self = this;
			$.each(options, function(key, value) { // This supports the 2 level depth that the options of dlePlayerVideo has. Would review if we ever need more depth.
				self._setOption(key, value);
			});

			return this;
		},
		_setOption: function(key, value) {
			var self = this;

			// The ability to set options is limited at this time.

			switch(key) {
				case "volume" :
					this.volume(value);
					break;
				case "muted" :
					this._muted(value);
					break;
				case "globalVolume" :
					this.options[key] = value;
					break;
				case "cssSelectorAncestor" :
					this._cssSelectorAncestor(value); // Set and refresh all associations for the new ancestor.
					break;
				case "cssSelector" :
					$.each(value, function(fn, cssSel) {
						self._cssSelector(fn, cssSel); // NB: The option is set inside this function, after further validity checks.
					});
					break;
				case "playbackRate" :
					this.options[key] = value = this._limitValue(value, this.options.minPlaybackRate, this.options.maxPlaybackRate);
					if(this.html.used) {
						this._html_setProperty('playbackRate', value);
					}
					this._updatePlaybackRate();
					break;
				case "defaultPlaybackRate" :
					this.options[key] = value = this._limitValue(value, this.options.minPlaybackRate, this.options.maxPlaybackRate);
					if(this.html.used) {
						this._html_setProperty('defaultPlaybackRate', value);
					}
					this._updatePlaybackRate();
					break;
				case "minPlaybackRate" :
					this.options[key] = value = this._limitValue(value, 0.1, this.options.maxPlaybackRate - 0.1);
					this._updatePlaybackRate();
					break;
				case "maxPlaybackRate" :
					this.options[key] = value = this._limitValue(value, this.options.minPlaybackRate + 0.1, 16);
					this._updatePlaybackRate();
					break;
				case "fullScreen" :
					if(this.options[key] !== value) { // if changed
						var wkv = $.dlePlayerVideo.nativeFeatures.fullscreen.used.webkitVideo;
						if(!wkv || wkv && !this.status.waitForPlay) {
							if(!wkv) { // No sensible way to unset option on these devices.
								this.options[key] = value;
							}
							if(value) {
								this._requestFullscreen();
							} else {
								this._exitFullscreen();
							}
							if(!wkv) {
								this._setOption("fullWindow", value);
							}
						}
					}
					break;
				case "fullWindow" :
					if(this.options[key] !== value) { // if changed
						this._removeUiClass();
						this.options[key] = value;
						this._refreshSize();
					}
					break;
				case "size" :
					if(!this.options.fullWindow && this.options[key].cssClass !== value.cssClass) {
						this._removeUiClass();
					}
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this._refreshSize();
					break;
				case "sizeFull" :
					if(this.options.fullWindow && this.options[key].cssClass !== value.cssClass) {
						this._removeUiClass();
					}
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this._refreshSize();
					break;
				case "autohide" :
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this._updateAutohide();
					break;
				case "loop" :
					this._loop(value);
					break;
				case "remainingDuration" :
					this.options[key] = value;
					this._updateInterface();
					break;
				case "toggleDuration" :
					this.options[key] = value;
					break;
				case "nativeVideoControls" :
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this.status.nativeVideoControls = this._uaBlocklist(this.options.nativeVideoControls);
					this._restrictNativeVideoControls();
					this._updateNativeVideoControls();
					break;
				case "noFullWindow" :
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this.status.nativeVideoControls = this._uaBlocklist(this.options.nativeVideoControls); // Need to check again as noFullWindow can depend on this flag and the restrict() can override it.
					this.status.noFullWindow = this._uaBlocklist(this.options.noFullWindow);
					this._restrictNativeVideoControls();
					this._updateButtons();
					break;
				case "noVolume" :
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					this.status.noVolume = this._uaBlocklist(this.options.noVolume);
					this._updateVolume();
					this._updateMute();
					break;
				case "emulateHtml" :
					if(this.options[key] !== value) { // To avoid multiple event handlers being created, if true already.
						this.options[key] = value;
						if(value) {
							this._emulateHtmlBridge();
						} else {
							this._destroyHtmlBridge();
						}
					}
					break;
				case "timeFormat" :
					this.options[key] = $.extend({}, this.options[key], value); // store a merged copy of it, incase not all properties changed.
					break;
				case "keyEnabled" :
					this.options[key] = value;
					if(!value && this === $.dlePlayerVideo.focus) {
						$.dlePlayerVideo.focus = null;
					}
					break;
				case "keyBindings" :
					this.options[key] = $.extend(true, {}, this.options[key], value); // store a merged DEEP copy of it, incase not all properties changed.
					break;
				case "audioFullScreen" :
					this.options[key] = value;
					break;
				case "autoBlur" :
					this.options[key] = value;
					break;
			}

			return this;
		},
		// End of: (Options code adapted from ui.widget.js)

		_refreshSize: function() {
			this._setSize(); // update status and dlePlayerVideo element size
			this._addUiClass(); // update the ui class
			this._updateSize(); // update internal sizes
			this._updateButtons();
			this._updateAutohide();
			this._trigger($.dlePlayerVideo.event.resize);
		},
		_setSize: function() {
			// Determine the current size from the options
			if(this.options.fullWindow) {
				this.status.width = this.options.sizeFull.width;
				this.status.height = this.options.sizeFull.height;
				this.status.cssClass = this.options.sizeFull.cssClass;
			} else {
				this.status.width = this.options.size.width;
				this.status.height = this.options.size.height;
				this.status.cssClass = this.options.size.cssClass;
			}

			// Set the size of the dlePlayerVideo area.
			this.element.css({'width': this.status.width, 'height': this.status.height});
		},
		_addUiClass: function() {
			if(this.ancestorJq.length) {
				this.ancestorJq.addClass(this.status.cssClass);
			}
		},
		_removeUiClass: function() {
			if(this.ancestorJq.length) {
				this.ancestorJq.removeClass(this.status.cssClass);
			}
		},
		_updateSize: function() {
			// The poster uses show/hide so can simply resize it.
			this.internal.poster.jq.css({'width': this.status.width, 'height': this.status.height});

			// Video html or flash resized if necessary at this time, or if native video controls being used.
			if(!this.status.waitForPlay && this.html.active && this.status.video || this.html.video.available && this.html.used && this.status.nativeVideoControls) {
				this.internal.video.jq.css({'width': this.status.width, 'height': this.status.height});
			}
			else if(!this.status.waitForPlay && this.flash.active && this.status.video) {
				this.internal.flash.jq.css({'width': this.status.width, 'height': this.status.height});
			}
		},
		_updateAutohide: function() {
			var	self = this,
				event = "mousemove.dlePlayerVideo",
				namespace = ".dlePlayerVideoAutohide",
				eventType = event + namespace,
				handler = function(event) {
					var moved = false,
						deltaX, deltaY;
					if(typeof self.internal.mouse !== "undefined") {
						//get the change from last position to this position
						deltaX = self.internal.mouse.x - event.pageX;
						deltaY = self.internal.mouse.y - event.pageY;
						moved = (Math.floor(deltaX) > 0) || (Math.floor(deltaY)>0);
					} else {
						moved = true;
					}
					// store current position for next method execution
					self.internal.mouse = {
							x : event.pageX,
							y : event.pageY
					};
					// if mouse has been actually moved, do the gui fadeIn/fadeOut
					if (moved) {
						self.css.jq.gui.fadeIn(self.options.autohide.fadeIn, function() {
							clearTimeout(self.internal.autohideId);
							self.internal.autohideId = setTimeout( function() {
								self.css.jq.gui.fadeOut(self.options.autohide.fadeOut);
							}, self.options.autohide.hold);
						});
					}
				};

			if(this.css.jq.gui.length) {

				// End animations first so that its callback is executed now.
				// Otherwise an in progress fadeIn animation still has the callback to fadeOut again.
				this.css.jq.gui.stop(true, true);

				// Removes the fadeOut operation from the fadeIn callback.
				clearTimeout(this.internal.autohideId);
				// undefine mouse
				delete this.internal.mouse;

				this.element.unbind(namespace);
				this.css.jq.gui.unbind(namespace);

				if(!this.status.nativeVideoControls) {
					if(this.options.fullWindow && this.options.autohide.full || !this.options.fullWindow && this.options.autohide.restored) {
						this.element.bind(eventType, handler);
						this.css.jq.gui.bind(eventType, handler);
						this.css.jq.gui.hide();
					} else {
						this.css.jq.gui.show();
					}
				} else {
					this.css.jq.gui.hide();
				}
			}
		},
		fullScreen: function(event) {
			var guiAction = typeof event === "object"; // Flags GUI click events so we know this was not a direct command, but an action taken by the user on the GUI.
			if(guiAction && this.options.useStateClassSkin && this.options.fullScreen) {
				this._setOption("fullScreen", false);
			} else {
				this._setOption("fullScreen", true);
			}
		},
		restoreScreen: function() {
			this._setOption("fullScreen", false);
		},
		_fullscreenAddEventListeners: function() {
			var self = this,
				fs = $.dlePlayerVideo.nativeFeatures.fullscreen;

			if(fs.api.fullscreenEnabled) {
				if(fs.event.fullscreenchange) {
					// Create the event handler function and store it for removal.
					if(typeof this.internal.fullscreenchangeHandler !== 'function') {
						this.internal.fullscreenchangeHandler = function() {
							self._fullscreenchange();
						};
					}
					document.addEventListener(fs.event.fullscreenchange, this.internal.fullscreenchangeHandler, false);
				}
				// No point creating handler for fullscreenerror.
				// Either logic avoids fullscreen occurring (w3c/moz), or their is no event on the browser (webkit).
			}
		},
		_fullscreenRemoveEventListeners: function() {
			var fs = $.dlePlayerVideo.nativeFeatures.fullscreen;
			if(this.internal.fullscreenchangeHandler) {
				document.removeEventListener(fs.event.fullscreenchange, this.internal.fullscreenchangeHandler, false);
			}
		},
		_fullscreenchange: function() {
			// If nothing is fullscreen, then we cannot be in fullscreen mode.
			if(this.options.fullScreen && !$.dlePlayerVideo.nativeFeatures.fullscreen.api.fullscreenElement()) {
				this._setOption("fullScreen", false);
			}
		},
		_requestFullscreen: function() {
			// Either the container or the dlePlayerVideo div
			var e = this.ancestorJq.length ? this.ancestorJq[0] : this.element[0],
				fs = $.dlePlayerVideo.nativeFeatures.fullscreen;

			// This method needs the video element. For iOS and Android.
			if(fs.used.webkitVideo) {
				e = this.htmlElement.video;
			}

			if(fs.api.fullscreenEnabled) {
				fs.api.requestFullscreen(e);
			}
		},
		_exitFullscreen: function() {

			var fs = $.dlePlayerVideo.nativeFeatures.fullscreen,
				e;

			// This method needs the video element. For iOS and Android.
			if(fs.used.webkitVideo) {
				e = this.htmlElement.video;
			}

			if(fs.api.fullscreenEnabled) {
				fs.api.exitFullscreen(e);
			}
		},
		_html_initMedia: function(media) {
			// Remove any existing track elements
			var $media = $(this.htmlElement.media).empty();

			// Create any track elements given with the media, as an Array of track Objects.
			$.each(media.track || [], function(i,v) {
				var track = document.createElement('track');
				track.setAttribute("kind", v.kind ? v.kind : "");
				track.setAttribute("src", v.src ? v.src : "");
				track.setAttribute("srclang", v.srclang ? v.srclang : "");
				track.setAttribute("label", v.label ? v.label : "");
				if(v.def) {
					track.setAttribute("default", v.def);
				}
				$media.append(track);
			});

			this.htmlElement.media.src = this.status.src;

			if(this.options.preload !== 'none') {
				this._html_load(); // See function for comments
			}
			this._trigger($.dlePlayerVideo.event.timeupdate); // The flash generates this event for its solution.
		},
		_html_setFormat: function(media) {
			var self = this;
			// Always finds a format due to checks in setMedia()
			$.each(this.formats, function(priority, format) {
				if(self.html.support[format] && media[format]) {
					self.status.src = media[format];
					self.status.format[format] = true;
					self.status.formatType = format;
					return false;
				}
			});
		},
		_html_setAudio: function(media) {
			this._html_setFormat(media);
			this.htmlElement.media = this.htmlElement.audio;
			this._html_initMedia(media);
		},
		_html_setVideo: function(media) {
			this._html_setFormat(media);
			if(this.status.nativeVideoControls) {
				this.htmlElement.video.poster = this._validString(media.poster) ? media.poster : "";
			}
			this.htmlElement.media = this.htmlElement.video;
			this._html_initMedia(media);
		},
		_html_resetMedia: function() {
			if(this.htmlElement.media) {
				if(this.htmlElement.media.id === this.internal.video.id && !this.status.nativeVideoControls) {
					this.internal.video.jq.css({'width':'0px', 'height':'0px'});
				}
				this.htmlElement.media.pause();
			}
		},
		_html_clearMedia: function() {
			if(this.htmlElement.media) {
				this.htmlElement.media.src = "about:blank";
				// The following load() is only required for Firefox 3.6 (PowerMacs).
				// Recent HTMl5 browsers only require the src change. Due to changes in W3C spec and load() effect.
				this.htmlElement.media.load(); // Stops an old, "in progress" download from continuing the download. Triggers the loadstart, error and emptied events, due to the empty src. Also an abort event if a download was in progress.
			}
		},
		_html_load: function() {
			// This function remains to allow the early HTML5 browsers to work, such as Firefox 3.6
			// A change in the W3C spec for the media.load() command means that this is no longer necessary.
			// This command should be removed and actually causes minor undesirable effects on some browsers. Such as loading the whole file and not only the metadata.
			if(this.status.waitForLoad) {
				this.status.waitForLoad = false;
				this.htmlElement.media.load();
			}
			clearTimeout(this.internal.htmlDlyCmdId);
		},
		_html_play: function(time) {
			var self = this,
				media = this.htmlElement.media;

			this.androidFix.pause = false; // Cancel the pause fix.

			this._html_load(); // Loads if required and clears any delayed commands.

			// Setup the Android Fix.
			if(this.androidFix.setMedia) {
				this.androidFix.play = true;
				this.androidFix.time = time;

			} else if(!isNaN(time)) {

				// Attempt to play it, since iOS has been ignoring commands
				if(this.internal.cmdsIgnored) {
					media.play();
				}

				try {
					// !media.seekable is for old HTML5 browsers, like Firefox 3.6.
					// Checking seekable.length is important for iOS6 to work with setMedia().play(time)
					if(!media.seekable || typeof media.seekable === "object" && media.seekable.length > 0) {
						media.currentTime = time;
						media.play();
					} else {
						throw 1;
					}
				} catch(err) {
					this.internal.htmlDlyCmdId = setTimeout(function() {
						self.play(time);
					}, 250);
					return; // Cancel execution and wait for the delayed command.
				}
			} else {
				media.play();
			}
			this._html_checkWaitForPlay();
		},
		_html_pause: function(time) {
			var self = this,
				media = this.htmlElement.media;

			this.androidFix.play = false; // Cancel the play fix.

			if(time > 0) { // We do not want the stop() command, which does pause(0), causing a load operation.
				this._html_load(); // Loads if required and clears any delayed commands.
			} else {
				clearTimeout(this.internal.htmlDlyCmdId);
			}

			// Order of these commands is important for Safari (Win) and IE9. Pause then change currentTime.
			media.pause();

			// Setup the Android Fix.
			if(this.androidFix.setMedia) {
				this.androidFix.pause = true;
				this.androidFix.time = time;

			} else if(!isNaN(time)) {
				try {
					if(!media.seekable || typeof media.seekable === "object" && media.seekable.length > 0) {
						media.currentTime = time;
					} else {
						throw 1;
					}
				} catch(err) {
					this.internal.htmlDlyCmdId = setTimeout(function() {
						self.pause(time);
					}, 250);
					return; // Cancel execution and wait for the delayed command.
				}
			}
			if(time > 0) { // Avoids a setMedia() followed by stop() or pause(0) hiding the video play button.
				this._html_checkWaitForPlay();
			}
		},
		_html_playHead: function(percent) {
			var self = this,
				media = this.htmlElement.media;

			this._html_load(); // Loads if required and clears any delayed commands.

			// This playHead() method needs a refactor to apply the android fix.

			try {
				if(typeof media.seekable === "object" && media.seekable.length > 0) {
					media.currentTime = percent * media.seekable.end(media.seekable.length-1) / 100;
				} else if(media.duration > 0 && !isNaN(media.duration)) {
					media.currentTime = percent * media.duration / 100;
				} else {
					throw "e";
				}
			} catch(err) {
				this.internal.htmlDlyCmdId = setTimeout(function() {
					self.playHead(percent);
				}, 250);
				return; // Cancel execution and wait for the delayed command.
			}
			if(!this.status.waitForLoad) {
				this._html_checkWaitForPlay();
			}
		},
		_html_checkWaitForPlay: function() {
			if(this.status.waitForPlay) {
				this.status.waitForPlay = false;
				if(this.css.jq.videoPlay.length) {
					this.css.jq.videoPlay.hide();
				}
				if(this.status.video) {
					this.internal.poster.jq.hide();
					this.internal.video.jq.css({'width': this.status.width, 'height': this.status.height});
				}
			}
		},
		_html_setProperty: function(property, value) {
			if(this.html.audio.available) {
				this.htmlElement.audio[property] = value;
			}
			if(this.html.video.available) {
				this.htmlElement.video[property] = value;
			}
		},
		_aurora_setAudio: function(media) {
			var self = this;

			// Always finds a format due to checks in setMedia()
			$.each(this.formats, function(priority, format) {
				if(self.aurora.support[format] && media[format]) {
					self.status.src = media[format];
					self.status.format[format] = true;
					self.status.formatType = format;

					return false;
				}
			});

			this.aurora.player = new AV.Player.fromURL(this.status.src);
			this._addAuroraEventListeners(this.aurora.player, this.aurora);

			if(this.options.preload === 'auto') {
				this._aurora_load();
				this.status.waitForLoad = false;
			}
		},
		_aurora_resetMedia: function() {
			if (this.aurora.player) {
				this.aurora.player.stop();
			}
		},
		_aurora_clearMedia: function() {
			// Nothing to clear.
		},
		_aurora_load: function() {
			if(this.status.waitForLoad) {
				this.status.waitForLoad = false;
				this.aurora.player.preload();
			}
		},
		_aurora_play: function(time) {
			if (!this.status.waitForLoad) {
				if (!isNaN(time)) {
					this.aurora.player.seek(time);
				}
			}
			if (!this.aurora.player.playing) {
				this.aurora.player.play();
			}
			this.status.waitForLoad = false;
			this._aurora_checkWaitForPlay();

			// No event from the player, update UI now.
			this._updateButtons(true);
			this._trigger($.dlePlayerVideo.event.play);
		},
		_aurora_pause: function(time) {
			if (!isNaN(time)) {
				this.aurora.player.seek(time * 1000);
			}
			this.aurora.player.pause();

			if(time > 0) { // Avoids a setMedia() followed by stop() or pause(0) hiding the video play button.
				this._aurora_checkWaitForPlay();
			}

			// No event from the player, update UI now.
			this._updateButtons(false);
			this._trigger($.dlePlayerVideo.event.pause);
		},
		_aurora_playHead: function(percent) {
			if(this.aurora.player.duration > 0) {
				// The seek() sould be in milliseconds, but the only codec that works with seek (aac.js) uses seconds.
				this.aurora.player.seek(percent * this.aurora.player.duration / 100); // Using seconds
			}

			if(!this.status.waitForLoad) {
				this._aurora_checkWaitForPlay();
			}
		},
		_aurora_checkWaitForPlay: function() {
			if(this.status.waitForPlay) {
				this.status.waitForPlay = false;
			}
		},
		_aurora_volume: function(v) {
			this.aurora.player.volume = v * 100;
		},
		_aurora_mute: function(m) {
			if (m) {
				this.aurora.properties.lastvolume = this.aurora.player.volume;
				this.aurora.player.volume = 0;
			} else {
				this.aurora.player.volume = this.aurora.properties.lastvolume;
			}
			this.aurora.properties.muted = m;
		},
		_flash_setAudio: function(media) {
			var self = this;
			try {
				// Always finds a format due to checks in setMedia()
				$.each(this.formats, function(priority, format) {
					if(self.flash.support[format] && media[format]) {
						switch (format) {
							case "m4a" :
							case "fla" :
								self._getMovie().fl_setAudio_m4a(media[format]);
								break;
							case "mp3" :
								self._getMovie().fl_setAudio_mp3(media[format]);
								break;
							case "rtmpa":
								self._getMovie().fl_setAudio_rtmp(media[format]);
								break;
						}
						self.status.src = media[format];
						self.status.format[format] = true;
						self.status.formatType = format;
						return false;
					}
				});

				if(this.options.preload === 'auto') {
					this._flash_load();
					this.status.waitForLoad = false;
				}
			} catch(err) { this._flashError(err); }
		},
		_flash_setVideo: function(media) {
			var self = this;
			try {
				// Always finds a format due to checks in setMedia()
				$.each(this.formats, function(priority, format) {
					if(self.flash.support[format] && media[format]) {
						switch (format) {
							case "m4v" :
							case "flv" :
								self._getMovie().fl_setVideo_m4v(media[format]);
								break;
							case "rtmpv":
								self._getMovie().fl_setVideo_rtmp(media[format]);
								break;
						}
						self.status.src = media[format];
						self.status.format[format] = true;
						self.status.formatType = format;
						return false;
					}
				});

				if(this.options.preload === 'auto') {
					this._flash_load();
					this.status.waitForLoad = false;
				}
			} catch(err) { this._flashError(err); }
		},
		_flash_resetMedia: function() {
			this.internal.flash.jq.css({'width':'0px', 'height':'0px'}); // Must do via CSS as setting attr() to zero causes a jQuery error in IE.
			this._flash_pause(NaN);
		},
		_flash_clearMedia: function() {
			try {
				this._getMovie().fl_clearMedia();
			} catch(err) { this._flashError(err); }
		},
		_flash_load: function() {
			try {
				this._getMovie().fl_load();
			} catch(err) { this._flashError(err); }
			this.status.waitForLoad = false;
		},
		_flash_play: function(time) {
			try {
				this._getMovie().fl_play(time);
			} catch(err) { this._flashError(err); }
			this.status.waitForLoad = false;
			this._flash_checkWaitForPlay();
		},
		_flash_pause: function(time) {
			try {
				this._getMovie().fl_pause(time);
			} catch(err) { this._flashError(err); }
			if(time > 0) { // Avoids a setMedia() followed by stop() or pause(0) hiding the video play button.
				this.status.waitForLoad = false;
				this._flash_checkWaitForPlay();
			}
		},
		_flash_playHead: function(p) {
			try {
				this._getMovie().fl_play_head(p);
			} catch(err) { this._flashError(err); }
			if(!this.status.waitForLoad) {
				this._flash_checkWaitForPlay();
			}
		},
		_flash_checkWaitForPlay: function() {
			if(this.status.waitForPlay) {
				this.status.waitForPlay = false;
				if(this.css.jq.videoPlay.length) {
					this.css.jq.videoPlay.hide();
				}
				if(this.status.video) {
					this.internal.poster.jq.hide();
					this.internal.flash.jq.css({'width': this.status.width, 'height': this.status.height});
				}
			}
		},
		_flash_volume: function(v) {
			try {
				this._getMovie().fl_volume(v);
			} catch(err) { this._flashError(err); }
		},
		_flash_mute: function(m) {
			try {
				this._getMovie().fl_mute(m);
			} catch(err) { this._flashError(err); }
		},
		_getMovie: function() {
			return document[this.internal.flash.id];
		},
		_getFlashPluginVersion: function() {

			// _getFlashPluginVersion() code influenced by:
			// - FlashReplace 1.01: http://code.google.com/p/flashreplace/
			// - SWFObject 2.2: http://code.google.com/p/swfobject/

			var version = 0,
				flash;
			if(window.ActiveXObject) {
				try {
					flash = new ActiveXObject("ShockwaveFlash.ShockwaveFlash");
					if (flash) { // flash will return null when ActiveX is disabled
						var v = flash.GetVariable("$version");
						if(v) {
							v = v.split(" ")[1].split(",");
							version = parseInt(v[0], 10) + "." + parseInt(v[1], 10);
						}
					}
				} catch(e) {}
			}
			else if(navigator.plugins && navigator.mimeTypes.length > 0) {
				flash = navigator.plugins["Shockwave Flash"];
				if(flash) {
					version = navigator.plugins["Shockwave Flash"].description.replace(/.*\s(\d+\.\d+).*/, "$1");
				}
			}
			return version * 1; // Converts to a number
		},
		_checkForFlash: function (version) {
			var flashOk = false;
			if(this._getFlashPluginVersion() >= version) {
				flashOk = true;
			}
			return flashOk;
		},
		_validString: function(url) {
			return (url && typeof url === "string"); // Empty strings return false
		},
		_limitValue: function(value, min, max) {
			return (value < min) ? min : ((value > max) ? max : value);
		},
		_urlNotSetError: function(context) {
			this._error( {
				type: $.dlePlayerVideo.error.URL_NOT_SET,
				context: context,
				message: $.dlePlayerVideo.errorMsg.URL_NOT_SET,
				hint: $.dlePlayerVideo.errorHint.URL_NOT_SET
			});
		},
		_flashError: function(error) {
			var errorType;
			if(!this.internal.ready) {
				errorType = "FLASH";
			} else {
				errorType = "FLASH_DISABLED";
			}
			this._error( {
				type: $.dlePlayerVideo.error[errorType],
				context: this.internal.flash.swf,
				message: $.dlePlayerVideo.errorMsg[errorType] + error.message,
				hint: $.dlePlayerVideo.errorHint[errorType]
			});
			// Allow the audio player to recover if display:none and then shown again, or with position:fixed on Firefox.
			// This really only affects audio in a media player, as an audio player could easily move the dlePlayerVideo element away from such issues.
			this.internal.flash.jq.css({'width':'1px', 'height':'1px'});
		},
		_error: function(error) {
			this._trigger($.dlePlayerVideo.event.error, error);
			if(this.options.errorAlerts) {
				this._alert("Error!" + (error.message ? "\n" + error.message : "") + (error.hint ? "\n" + error.hint : "") + "\nContext: " + error.context);
			}
		},
		_warning: function(warning) {
			this._trigger($.dlePlayerVideo.event.warning, undefined, warning);
			if(this.options.warningAlerts) {
				this._alert("Warning!" + (warning.message ? "\n" + warning.message : "") + (warning.hint ? "\n" + warning.hint : "") + "\nContext: " + warning.context);
			}
		},
		_alert: function(message) {
			var msg = "dlePlayerVideo " + this.version.script + " : id='" + this.internal.self.id +"' : " + message;
			if(!this.options.consoleAlerts) {
				alert(msg);
			} else if(window.console && window.console.log) {
				window.console.log(msg);
			}
		},
		_emulateHtmlBridge: function() {
			var self = this;

			// Emulate methods on dlePlayerVideo's DOM element.
			$.each( $.dlePlayerVideo.emulateMethods.split(/\s+/g), function(i, name) {
				self.internal.domNode[name] = function(arg) {
					self[name](arg);
				};

			});

			// Bubble dlePlayerVideo events to its DOM element.
			$.each($.dlePlayerVideo.event, function(eventName,eventType) {
				var nativeEvent = true;
				$.each( $.dlePlayerVideo.reservedEvent.split(/\s+/g), function(i, name) {
					if(name === eventName) {
						nativeEvent = false;
						return false;
					}
				});
				if(nativeEvent) {
					self.element.bind(eventType + ".dlePlayerVideo.dlePlayerVideoHtml", function() { // With .dlePlayerVideo & .dlePlayerVideoHtml namespaces.
						self._emulateHtmlUpdate();
						var domEvent = document.createEvent("Event");
						domEvent.initEvent(eventName, false, true);
						self.internal.domNode.dispatchEvent(domEvent);
					});
				}
				// The error event would require a special case
			});

			// IE9 has a readyState property on all elements. The document should have it, but all (except media) elements inherit it in IE9. This conflicts with Popcorn, which polls the readyState.
		},
		_emulateHtmlUpdate: function() {
			var self = this;

			$.each( $.dlePlayerVideo.emulateStatus.split(/\s+/g), function(i, name) {
				self.internal.domNode[name] = self.status[name];
			});
			$.each( $.dlePlayerVideo.emulateOptions.split(/\s+/g), function(i, name) {
				self.internal.domNode[name] = self.options[name];
			});
		},
		_destroyHtmlBridge: function() {
			var self = this;

			// Bridge event handlers are also removed by destroy() through .dlePlayerVideo namespace.
			this.element.unbind(".dlePlayerVideoHtml"); // Remove all event handlers created by the dlePlayerVideo bridge. So you can change the emulateHtml option.

			// Remove the methods and properties
			var emulated = $.dlePlayerVideo.emulateMethods + " " + $.dlePlayerVideo.emulateStatus + " " + $.dlePlayerVideo.emulateOptions;
			$.each( emulated.split(/\s+/g), function(i, name) {
				delete self.internal.domNode[name];
			});
		}
	};

	$.dlePlayerVideo.error = {
		FLASH: "e_flash",
		FLASH_DISABLED: "e_flash_disabled",
		NO_SOLUTION: "e_no_solution",
		NO_SUPPORT: "e_no_support",
		URL: "e_url",
		URL_NOT_SET: "e_url_not_set",
		VERSION: "e_version"
	};

	$.dlePlayerVideo.errorMsg = {
		FLASH: "dlePlayerVideo's Flash fallback is not configured correctly, or a command was issued before the dlePlayerVideo Ready event. Details: ", // Used in: _flashError()
		FLASH_DISABLED: "dlePlayerVideo's Flash fallback has been disabled by the browser due to the CSS rules you have used. Details: ", // Used in: _flashError()
		NO_SOLUTION: "No solution can be found by dlePlayerVideo in this browser. Neither HTML nor Flash can be used.", // Used in: _init()
		NO_SUPPORT: "It is not possible to play any media format provided in setMedia() on this browser using your current options.", // Used in: setMedia()
		URL: "Media URL could not be loaded.", // Used in: dlePlayerVideoFlashEvent() and _addHtmlEventListeners()
		URL_NOT_SET: "Attempt to issue media playback commands, while no media url is set.", // Used in: load(), play(), pause(), stop() and playHead()
		VERSION: "dlePlayerVideo " + $.dlePlayerVideo.prototype.version.script + " needs Jplayer.swf version " + $.dlePlayerVideo.prototype.version.needFlash + " but found " // Used in: dlePlayerVideoReady()
	};

	$.dlePlayerVideo.errorHint = {
		FLASH: "Check your swfPath option and that Jplayer.swf is there.",
		FLASH_DISABLED: "Check that you have not display:none; the dlePlayerVideo entity or any ancestor.",
		NO_SOLUTION: "Review the dlePlayerVideo options: support and supplied.",
		NO_SUPPORT: "Video or audio formats defined in the supplied option are missing.",
		URL: "Check media URL is valid.",
		URL_NOT_SET: "Use setMedia() to set the media URL.",
		VERSION: "Update dlePlayerVideo files."
	};

	$.dlePlayerVideo.warning = {
		CSS_SELECTOR_COUNT: "e_css_selector_count",
		CSS_SELECTOR_METHOD: "e_css_selector_method",
		CSS_SELECTOR_STRING: "e_css_selector_string",
		OPTION_KEY: "e_option_key"
	};

	$.dlePlayerVideo.warningMsg = {
		CSS_SELECTOR_COUNT: "The number of css selectors found did not equal one: ",
		CSS_SELECTOR_METHOD: "The methodName given in dlePlayerVideo('cssSelector') is not a valid dlePlayerVideo method.",
		CSS_SELECTOR_STRING: "The methodCssSelector given in dlePlayerVideo('cssSelector') is not a String or is empty.",
		OPTION_KEY: "The option requested in dlePlayerVideo('option') is undefined."
	};

	$.dlePlayerVideo.warningHint = {
		CSS_SELECTOR_COUNT: "Check your css selector and the ancestor.",
		CSS_SELECTOR_METHOD: "Check your method name.",
		CSS_SELECTOR_STRING: "Check your css selector is a string.",
		OPTION_KEY: "Check your option name."
	};

}));




/*
 * Playlist Object for the dlePlayerVideo Plugin
 * http://www.jplayer.org
 *
 * Copyright (c) 2009 - 2014 Happyworm Ltd
 * Licensed under the MIT license.
 * http://www.opensource.org/licenses/MIT
 *
 * Author: Mark J Panaghiston
 * Version: 2.4.1
 * Date: 19th November 2014
 *
 * Requires:
 *  - jQuery 1.7.0+
 *  - dlePlayerVideo 2.8.2+
 */

/*global dlePlayerVideoPlaylist:true */

(function($, undefined) {

	dlePlayerVideoPlaylist = function(cssSelector, playlist, options) {
		var self = this;

		this.current = 0;
		this.loop = false; // Flag used with the dlePlayerVideo repeat event
		this.shuffled = false;
		this.removing = false; // Flag is true during remove animation, disabling the remove() method until complete.

		this.cssSelector = $.extend({}, this._cssSelector, cssSelector); // Object: Containing the css selectors for dlePlayerVideo and its cssSelectorAncestor
		this.options = $.extend(true, {
			keyBindings: {
				next: {
					key: 221, // ]
					fn: function() {
						self.next();
					}
				},
				previous: {
					key: 219, // [
					fn: function() {
						self.previous();
					}
				},
				shuffle: {
					key: 83, // s
					fn: function() {
						self.shuffle();
					}
				}
			},
			stateClass: {
				shuffled: "mdtc-clnplrv-state-shuffled"
			}
		}, this._options, options); // Object: The dlePlayerVideo constructor options for this playlist and the playlist options

		this.playlist = []; // Array of Objects: The current playlist displayed (Un-shuffled or Shuffled)
		this.original = []; // Array of Objects: The original playlist

		this._initPlaylist(playlist); // Copies playlist to this.original. Then mirrors this.original to this.playlist. Creating two arrays, where the element pointers match. (Enables pointer comparison.)

		// Setup the css selectors for the extra interface items used by the playlist.
		this.cssSelector.details = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-details"; // Note that dlePlayerVideo controls the text in the title element.
		this.cssSelector.playlist = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-playlist";
		this.cssSelector.next = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-next";
		this.cssSelector.previous = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-previous";
		this.cssSelector.shuffle = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-shuffle";
		this.cssSelector.shuffleOff = this.cssSelector.cssSelectorAncestor + " .mdtc-clnplrv-shuffle-off";

		// Override the cssSelectorAncestor given in options
		this.options.cssSelectorAncestor = this.cssSelector.cssSelectorAncestor;

		// Override the default repeat event handler
		this.options.repeat = function(event) {
			self.loop = event.dlePlayerVideo.options.loop;
		};

		// Create a ready event handler to initialize the playlist
		$(this.cssSelector.dlePlayerVideo).bind($.dlePlayerVideo.event.ready, function() {
			self._init();
		});

		// Create an ended event handler to move to the next item
		$(this.cssSelector.dlePlayerVideo).bind($.dlePlayerVideo.event.ended, function() {
			self.next();
		});

		// Create a play event handler to pause other instances
		$(this.cssSelector.dlePlayerVideo).bind($.dlePlayerVideo.event.play, function() {
			$(this).dlePlayerVideo("pauseOthers");
		});

		// Create a resize event handler to show the title in full screen mode.
		$(this.cssSelector.dlePlayerVideo).bind($.dlePlayerVideo.event.resize, function(event) {
			if(event.dlePlayerVideo.options.fullScreen) {
				$(self.cssSelector.details).show();
			} else {
				$(self.cssSelector.details).hide();
			}
		});

		// Create click handlers for the extra buttons that do playlist functions.
		$(this.cssSelector.previous).click(function(e) {
			e.preventDefault();
			self.previous();
			self.blur(this);
		});

		$(this.cssSelector.next).click(function(e) {
			e.preventDefault();
			self.next();
			self.blur(this);
		});

		$(this.cssSelector.shuffle).click(function(e) {
			e.preventDefault();
			if(self.shuffled && $(self.cssSelector.dlePlayerVideo).dlePlayerVideo("option", "useStateClassSkin")) {
				self.shuffle(false);
			} else {
				self.shuffle(true);
			}
			self.blur(this);
		});
		$(this.cssSelector.shuffleOff).click(function(e) {
			e.preventDefault();
			self.shuffle(false);
			self.blur(this);
		}).hide();

		// Put the title in its initial display state
		if(!this.options.fullScreen) {
			$(this.cssSelector.details).hide();
		}

		// Remove the empty <li> from the page HTML. Allows page to be valid HTML, while not interfereing with display animations
		$(this.cssSelector.playlist + " ul").empty();

		// Create .on() handlers for the playlist items along with the free media and remove controls.
		this._createItemHandlers();

		// Instance dlePlayerVideo
		$(this.cssSelector.dlePlayerVideo).dlePlayerVideo(this.options);
	};

	dlePlayerVideoPlaylist.prototype = {
		_cssSelector: { // static object, instanced in constructor
			dlePlayerVideo: "#jquery_jplayer_1",
			cssSelectorAncestor: "#jp_video_container_1"
		},
		_options: { // static object, instanced in constructor
			playlistOptions: {
				autoPlay: false,
				loopOnPrevious: false,
				shuffleOnLoop: true,
				enableRemoveControls: false,
				displayTime: 'slow',
				addTime: 'fast',
				removeTime: 'fast',
				shuffleTime: 'slow',
				itemClass: "mdtc-clnplrv-playlist-item",
				freeGroupClass: "mdtc-clnplrv-free-media",
				freeItemClass: "mdtc-clnplrv-playlist-item-free",
				removeItemClass: "mdtc-clnplrv-playlist-item-remove"
			}
		},
		option: function(option, value) { // For changing playlist options only
			if(value === undefined) {
				return this.options.playlistOptions[option];
			}

			this.options.playlistOptions[option] = value;

			switch(option) {
				case "enableRemoveControls":
					this._updateControls();
					break;
				case "itemClass":
				case "freeGroupClass":
				case "freeItemClass":
				case "removeItemClass":
					this._refresh(true); // Instant
					this._createItemHandlers();
					break;
			}
			return this;
		},
		_init: function() {
			var self = this;
			this._refresh(function() {
				if(self.options.playlistOptions.autoPlay) {
					self.play(self.current);
				} else {
					self.select(self.current);
				}
			});
		},
		_initPlaylist: function(playlist) {
			this.current = 0;
			this.shuffled = false;
			this.removing = false;
			this.original = $.extend(true, [], playlist); // Copy the Array of Objects
			this._originalPlaylist();
		},
		_originalPlaylist: function() {
			var self = this;
			this.playlist = [];
			// Make both arrays point to the same object elements. Gives us 2 different arrays, each pointing to the same actual object. ie., Not copies of the object.
			$.each(this.original, function(i) {
				self.playlist[i] = self.original[i];
			});
		},
		_refresh: function(instant) {
			/* instant: Can be undefined, true or a function.
			 *	undefined -> use animation timings
			 *	true -> no animation
			 *	function -> use animation timings and excute function at half way point.
			 */
			var self = this;

			if(instant && !$.isFunction(instant)) {
				$(this.cssSelector.playlist + " ul").empty();
				$.each(this.playlist, function(i) {
					$(self.cssSelector.playlist + " ul").append(self._createListItem(self.playlist[i]));
				});
				this._updateControls();
			} else {
				var displayTime = $(this.cssSelector.playlist + " ul").children().length ? this.options.playlistOptions.displayTime : 0;

				$(this.cssSelector.playlist + " ul").slideUp(displayTime, function() {
					var $this = $(this);
					$(this).empty();

					$.each(self.playlist, function(i) {
						$this.append(self._createListItem(self.playlist[i]));
					});
					self._updateControls();
					if($.isFunction(instant)) {
						instant();
					}
					if(self.playlist.length) {
						$(this).slideDown(self.options.playlistOptions.displayTime);
					} else {
						$(this).show();
					}
				});
			}
		},
		_createListItem: function(media) {
			var self = this;

			// Wrap the <li> contents in a <div>
			var listItem = "<li><div>";

			// Create remove control
			listItem += "<a href='javascript:;' class='" + this.options.playlistOptions.removeItemClass + "'>&times;</a>";

			// Create links to free media
			if(media.free) {
				var first = true;
				listItem += "<span class='" + this.options.playlistOptions.freeGroupClass + "'>(";
				$.each(media, function(property,value) {
					if($.dlePlayerVideo.prototype.format[property]) { // Check property is a media format.
						if(first) {
							first = false;
						} else {
							listItem += " | ";
						}
						listItem += "<a class='" + self.options.playlistOptions.freeItemClass + "' href='" + value + "' tabindex='-1'>" + property + "</a>";
					}
				});
				listItem += ")</span>";
			}

			// The title is given next in the HTML otherwise the float:right on the free media corrupts in IE6/7
			listItem += "<span href='javascript:;' class='" + this.options.playlistOptions.itemClass + "' tabindex='0'>" + media.title + (media.artist ? " <span class='mdtc-clnplrv-artist'>by " + media.artist + "</span>" : "") + "</span>";
			listItem += "</div></li>";

			return listItem;
		},
		_createItemHandlers: function() {
			var self = this;
			// Create live handlers for the playlist items
			$(this.cssSelector.playlist).off("click", "span." + this.options.playlistOptions.itemClass).on("click", "span." + this.options.playlistOptions.itemClass, function(e) {
				e.preventDefault();
				var index = $(this).parent().parent().index();
				if(self.current !== index) {
					self.play(index);
				} else {
					$(self.cssSelector.dlePlayerVideo).dlePlayerVideo("play");
				}
				self.blur(this);
			});

			// Create live handlers that disable free media links to force access via right click
			$(this.cssSelector.playlist).off("click", "a." + this.options.playlistOptions.freeItemClass).on("click", "a." + this.options.playlistOptions.freeItemClass, function(e) {
				e.preventDefault();
				$(this).parent().parent().find("." + self.options.playlistOptions.itemClass).click();
				self.blur(this);
			});

			// Create live handlers for the remove controls
			$(this.cssSelector.playlist).off("click", "a." + this.options.playlistOptions.removeItemClass).on("click", "a." + this.options.playlistOptions.removeItemClass, function(e) {
				e.preventDefault();
				var index = $(this).parent().parent().index();
				self.remove(index);
				self.blur(this);
			});
		},
		_updateControls: function() {
			if(this.options.playlistOptions.enableRemoveControls) {
				$(this.cssSelector.playlist + " ." + this.options.playlistOptions.removeItemClass).show();
			} else {
				$(this.cssSelector.playlist + " ." + this.options.playlistOptions.removeItemClass).hide();
			}

			if(this.shuffled) {
				$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("addStateClass", "shuffled");
			} else {
				$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("removeStateClass", "shuffled");
			}
			if($(this.cssSelector.shuffle).length && $(this.cssSelector.shuffleOff).length) {
				if(this.shuffled) {
					$(this.cssSelector.shuffleOff).show();
					$(this.cssSelector.shuffle).hide();
				} else {
					$(this.cssSelector.shuffleOff).hide();
					$(this.cssSelector.shuffle).show();
				}
			}
		},
		_highlight: function(index) {
			if(this.playlist.length && index !== undefined) {
				$(this.cssSelector.playlist + " .mdtc-clnplrv-playlist-current").removeClass("mdtc-clnplrv-playlist-current");
				$(this.cssSelector.playlist + " li:nth-child(" + (index + 1) + ")").addClass("mdtc-clnplrv-playlist-current").find(".mdtc-clnplrv-playlist-item").addClass("mdtc-clnplrv-playlist-current");
				// $(this.cssSelector.details + " li").html("<span class='mdtc-clnplrv-title'>" + this.playlist[index].title + "</span>" + (this.playlist[index].artist ? " <span class='mdtc-clnplrv-artist'>by " + this.playlist[index].artist + "</span>" : ""));
			}
		},
		setPlaylist: function(playlist) {
			this._initPlaylist(playlist);
			this._init();
		},
		add: function(media, playNow) {
			$(this.cssSelector.playlist + " ul").append(this._createListItem(media)).find("li:last-child").hide().slideDown(this.options.playlistOptions.addTime);
			this._updateControls();
			this.original.push(media);
			this.playlist.push(media); // Both array elements share the same object pointer. Comforms with _initPlaylist(p) system.

			if(playNow) {
				this.play(this.playlist.length - 1);
			} else {
				if(this.original.length === 1) {
					this.select(0);
				}
			}
		},
		remove: function(index) {
			var self = this;

			if(index === undefined) {
				this._initPlaylist([]);
				this._refresh(function() {
					$(self.cssSelector.dlePlayerVideo).dlePlayerVideo("clearMedia");
				});
				return true;
			} else {

				if(this.removing) {
					return false;
				} else {
					index = (index < 0) ? self.original.length + index : index; // Negative index relates to end of array.
					if(0 <= index && index < this.playlist.length) {
						this.removing = true;

						$(this.cssSelector.playlist + " li:nth-child(" + (index + 1) + ")").slideUp(this.options.playlistOptions.removeTime, function() {
							$(this).remove();

							if(self.shuffled) {
								var item = self.playlist[index];
								$.each(self.original, function(i) {
									if(self.original[i] === item) {
										self.original.splice(i, 1);
										return false; // Exit $.each
									}
								});
								self.playlist.splice(index, 1);
							} else {
								self.original.splice(index, 1);
								self.playlist.splice(index, 1);
							}

							if(self.original.length) {
								if(index === self.current) {
									self.current = (index < self.original.length) ? self.current : self.original.length - 1; // To cope when last element being selected when it was removed
									self.select(self.current);
								} else if(index < self.current) {
									self.current--;
								}
							} else {
								$(self.cssSelector.dlePlayerVideo).dlePlayerVideo("clearMedia");
								self.current = 0;
								self.shuffled = false;
								self._updateControls();
							}

							self.removing = false;
						});
					}
					return true;
				}
			}
		},
		select: function(index) {
			index = (index < 0) ? this.original.length + index : index; // Negative index relates to end of array.
			if(0 <= index && index < this.playlist.length) {
				this.current = index;
				this._highlight(index);
				$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("setMedia", this.playlist[this.current]);
			} else {
				this.current = 0;
			}
		},
		play: function(index) {
			index = (index < 0) ? this.original.length + index : index; // Negative index relates to end of array.
			if(0 <= index && index < this.playlist.length) {
				if(this.playlist.length) {
					this.select(index);
					$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("play");
				}
			} else if(index === undefined) {
				$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("play");
			}
		},
		pause: function() {
			$(this.cssSelector.dlePlayerVideo).dlePlayerVideo("pause");
		},
		next: function() {
			var index = (this.current + 1 < this.playlist.length) ? this.current + 1 : 0;

			if(this.loop) {
				// See if we need to shuffle before looping to start, and only shuffle if more than 1 item.
				if(index === 0 && this.shuffled && this.options.playlistOptions.shuffleOnLoop && this.playlist.length > 1) {
					this.shuffle(true, true); // playNow
				} else {
					this.play(index);
				}
			} else {
				// The index will be zero if it just looped round
				if(index > 0) {
					this.play(index);
				}
			}
		},
		previous: function() {
			var index = (this.current - 1 >= 0) ? this.current - 1 : this.playlist.length - 1;

			if(this.loop && this.options.playlistOptions.loopOnPrevious || index < this.playlist.length - 1) {
				this.play(index);
			}
		},
		shuffle: function(shuffled, playNow) {
			var self = this;

			if(shuffled === undefined) {
				shuffled = !this.shuffled;
			}

			if(shuffled || shuffled !== this.shuffled) {

				$(this.cssSelector.playlist + " ul").slideUp(this.options.playlistOptions.shuffleTime, function() {
					self.shuffled = shuffled;
					if(shuffled) {
						self.playlist.sort(function() {
							return 0.5 - Math.random();
						});
					} else {
						self._originalPlaylist();
					}
					self._refresh(true); // Instant

					if(playNow || !$(self.cssSelector.dlePlayerVideo).data("dlePlayerVideo").status.paused) {
						self.play(0);
					} else {
						self.select(0);
					}

					$(this).slideDown(self.options.playlistOptions.shuffleTime);
				});
			}
		},
		blur: function(that) {
			if($(this.cssSelector.dlePlayerVideo).dlePlayerVideo("option", "autoBlur")) {
				$(that).blur();
			}
		}
	};
})(jQuery);

(function ($) {
    var interval;
    function marquee(that) {
        function slide() {
            var container   = that.next(),
                innerWidth  = container.find('.mdtc-clnplrv-song-title-inner'),
                mOuterWidth = container.find('.mdtc-clnplrv-song-title').width(),
                mInnerWidth = innerWidth.width(),
                diffMargin  = mInnerWidth - mOuterWidth;

            if (diffMargin > 0) {
                innerWidth.animate({marginLeft: '-=' + (diffMargin - 10)}, 2500);
                setTimeout(function(){
                    innerWidth.animate({marginLeft: '+=' + (diffMargin - 10)}, 2500);
                }, 4000);
            }
        }
        slide();
        interval = setInterval(function(){
            slide();
        }, 8000);
    }

	function makeId() {
		var text = '',
			possible = 'abcdefghijklmnopqrstuvwxyz0123456789';

		for( var i=0; i < 15; i++ )
			text += possible.charAt(Math.floor(Math.random() * possible.length));

		return text;
	}

    var playerTemplate = '<div class="mdtc-clnplrv-audio mdtc-clnplrv-main-player">' +
            '<div class="mdtc-clnplrv-jplayer"><!-- --></div>' +
            '<div class="mdtc-clnplrv-type-playlist">' +
                '<div class="mdtc-clnplrv-gui mdtc-clnplrv-interface">' +
                    '<ul class="mdtc-clnplrv-controls">' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-previous mdtc-clnplrv-icon-prev" tabindex="1"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-play mdtc-clnplrv-icon-play" tabindex="1"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-pause mdtc-clnplrv-icon-pause" tabindex="1" style="display:none"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-next mdtc-clnplrv-icon-next" tabindex="1"></a></li>' +
                    '</ul>' +
                    '<div class="mdtc-clnplrv-seek-container">' +
                        '<div class="mdtc-clnplrv-seek-container-inner">' +
                            '<div class="mdtc-clnplrv-time">' +
                                '<div class="mdtc-clnplrv-current-time">00:00</div>' +
                                '<div class="mdtc-clnplrv-duration" style="display:none">00:00</div>' +
                            '</div>' +
                            '<div class="mdtc-clnplrv-song-title">' +
                                '<div class="mdtc-clnplrv-song-title-inner">' +
                                    '<strong id="current-song"><!-- --></strong>' +
                                '</div>' +
                            '</div>' +
                            '<div class="mdtc-clnplrv-progress">' +
                                '<div class="mdtc-clnplrv-seek-bar" style="width:0%">' +
                                    '<div class="mdtc-clnplrv-play-bar" style="width:0%"><!-- --></div>' +
                                '</div>' +
                            '</div>' +
                        '</div>' +
                    '</div>' +
					'<div class="mdtc-clnplrv-volume-bar">' +
						'<div class="mdtc-clnplrv-volume-bar-value"><i><!-- --></i></div>' +
					'</div>' +
					'<ul class="mdtc-clnplrv-toggles">' +
						'<li><a title="Repeat" tabindex="1" class="mdtc-clnplrv-repeat mdtc-clnplrv-icon-repeat" href="javascript:;"></a></li>' +
						'<li><a title="Repeat off" tabindex="1" class="mdtc-clnplrv-repeat-off mdtc-clnplrv-icon-repeat" href="javascript:;" style="display:none"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-shuffle mdtc-clnplrv-icon-shuffle" tabindex="1" title="Shuffle"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-shuffle-off mdtc-clnplrv-icon-shuffle" tabindex="1" title="Shuffle off" style="display:none"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-download-one mdtc-clnplrv-icon-download" tabindex="1" title="Download" style="display:none"></a></li>' +
					'</ul>' +
                '</div>' +
                '<div class="mdtc-clnplrv-playlist">' +
                    '<ul></ul>' +
                '</div>' +
                '<div class="mdtc-clnplrv-no-solution" style="display:none">' +
                    '<span>Update Required </span>' +
                    'To play the media you will need to either update your browser to a recent version' +
                '</div>' +
            '</div>' +
        '</div>';

    $.fn.cleanaudioplayer = function(options) {

        (function(down,  dlePlayerVideo, fn, $target){

            $('body').on('mousemove', '.mdtc-clnplrv-volume-bar', function(e){
                dlePlayerVideo && dlePlayerVideo[fn]($.extend({}, e, {currentTarget: $target}));
            }).on('mouseup', function(){
                dlePlayerVideo = false;
            }).on('mousedown', function(e){
                var $nodes = $(e.target).parents().add(e.currentTarget),
                    $volume = $nodes.filter('.mdtc-clnplrv-volume-bar').first(),
                    $seek = $nodes.filter('.mdtc-clnplrv-seek-bar').first();

                if ($seek.length) {
                    fn = 'seekBar';
                    $target = $seek;
                } else if ($volume.length) {
                    fn = 'volumeBar';
                    $target = $volume;
                }

                var $audio = $target && $target.closest('.mdtc-clnplrv-audio');
                dlePlayerVideo = $audio && $audio.children().first().add($audio.prev()).filter(function(){
                        return $(this).data('dlePlayerVideo');
                    }).first().data('dlePlayerVideo');
            });

        })();

        return this.each(function(index) {

			var settings = $.extend({
				theme: 'default',
				supplied: 'mp3,m4a,oga,wav',
				disableControls: false,
				autoplay: false,
				playlistHeight: false,
				rssUrl: false,
				playlist: false
			}, options );

			var soloDownload = false,
				soloDownloadUrl = '';

			if ($(this).find('> ul').length > 0) {

				var ul = $(this).find('> ul');

				ul.data('theme') ? settings.theme = ul.data('theme') : void(0);
				ul.data('supplied') ? settings.supplied = ul.data('supplied') : void(0);
				ul.data('disable-controls') ? settings.disableControls = ul.data('disable-controls') : void(0);
				ul.data('autoplay') ? settings.autoplay = ul.data('autoplay') : void(0);
				ul.data('playlist-height') ? settings.playlistHeight = ul.data('playlist-height') : void(0);
				ul.data('preload') ? settings.preload = ul.data('preload') : void(0);

				if ($(this).find('> ul li').length > 0) {
					var itemsArray = [];
					$(this).find('> ul li').each(function(){
						var li		= $(this),
							free	= li.data('free') ? true : false,
							type 	= li.data('type'),
							obj		= {title: li.data('title'), artist: li.data('artist'), free: free};

							obj[type] = li.data('url');

						itemsArray.push(obj);
					});

					if ($(this).find('> ul li').length == 1 && $(this).find('> ul li').data('free')) {
						soloDownload = true;
						soloDownloadUrl = $(this).find('> ul li').data('url');
					}

					settings.playlist = itemsArray;
				}
			}

			var currentId = makeId();
            $(this).html(playerTemplate);
            $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-cleanaudioplayer-id-' + currentId);

			if (soloDownload) {
				$(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-solo-download');
				$(this).find('.mdtc-clnplrv-download-one').attr('href', soloDownloadUrl);
			}

            if (settings.theme == 'dark') {
                $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-dark-theme');
            }

            if (settings.disableControls) {
                $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-disabled-controls');
            }

            if (settings.playlistHeight) {
                $(this).find('.mdtc-clnplrv-playlist > ul').css({
                    maxHeight: settings.playlistHeight,
                    overflowY: 'auto'
                });
            }

            if (settings.playlist.length > 1) {
                playerOptions = {
                    solution: 'html',
                    supplied: settings.supplied,
                    preload: settings.preload,
                    wmode: 'window',
                    ready: function (event) {
                        var that = $(this);
                        that.next().find('.mdtc-clnplrv-time').on('click', function () {
                            that.next().find('.mdtc-clnplrv-current-time, .mdtc-clnplrv-duration').toggle();
                        });
                        if (settings.autoplay) {
                            $(this).dlePlayerVideo('play');
                        };
						function resizeView() {
							if (that.next().is(':visible')) {
								var width = that.parent().width();
								if (width < 450) {
									that.parent().addClass('mdtc-clnplrv-responsive');
								} else {
									that.parent().removeClass('mdtc-clnplrv-responsive');
								}
							}
						};
						$(window).on('resize', function() {
							resizeView();
						});
						resizeView();
						setInterval(function(){
							resizeView();
						}, 250)
						marquee(that);
                    },
                    play: function (event) {
                        $(this).dlePlayerVideo("pauseOthers");
                    },
                    cssSelector: {
                        title: '#current-song'
                    },
					playlistOptions: {
						shuffleTime: 150
					}
                };

                var playerListFiles = settings.playlist;

                var playerSelectors = {
                    dlePlayerVideo: '.mdtc-clnplrv-cleanaudioplayer-id-' + currentId + ' .mdtc-clnplrv-jplayer',
                    cssSelectorAncestor: '.mdtc-clnplrv-cleanaudioplayer-id-' + currentId
                };

                var initPlayerAudio = new dlePlayerVideoPlaylist(playerSelectors, playerListFiles, playerOptions);

            } else {
                $('.mdtc-clnplrv-cleanaudioplayer-id-' + currentId + ' .mdtc-clnplrv-jplayer').dlePlayerVideo({
                    ready: function (event) {
                        $(this).dlePlayerVideo('setMedia', settings.playlist[0]);
                        var that = $(this);
                        that.next().find('.mdtc-clnplrv-time').on('click', function(){
                            that.next().find('.mdtc-clnplrv-current-time, .mdtc-clnplrv-duration').toggle();
                        });
                        if (settings.autoplay) {
                            $(this).dlePlayerVideo('play');
                        };
                        $(this).parent().addClass('mdtc-clnplrv-disabled-controls mdtc-clnplrv-solo-player')
						function resizeView() {
							if (that.next().is(':visible')) {
								var width	= that.parent().width();
								if (width < 450) {
									that.parent().addClass('mdtc-clnplrv-responsive');
								} else {
									that.parent().removeClass('mdtc-clnplrv-responsive');
								}
							}
						};
						$(window).on('resize', function() {
							resizeView();
						});
						resizeView();
						setInterval(function(){
							resizeView();
						}, 250)
						marquee(that);
                    },
                    play: function (event) {
                        $(this).dlePlayerVideo('pauseOthers');
                    },
                    cssSelector: {
                        title: '#current-song'
                    },
                    cssSelectorAncestor: '.mdtc-clnplrv-cleanaudioplayer-id-' + currentId,
                    supplied: settings.supplied
                });
            }

        });
    };

}(jQuery));





(function ($) {
    var interval;
    function marquee(that) {
        function slide() {
            var container   = that.next(),
                innerWidth  = container.find('.mdtc-clnplrv-song-title-inner'),
                mOuterWidth = container.find('.mdtc-clnplrv-song-title').width(),
                mInnerWidth = innerWidth.width(),
                diffMargin  = mInnerWidth - mOuterWidth;

            if (diffMargin > 0) {
                innerWidth.animate({marginLeft: '-=' + (diffMargin - 10)}, 2500);
                setTimeout(function(){
                    innerWidth.animate({marginLeft: '+=' + (diffMargin - 10)}, 2500);
                }, 4000);
            }
        }
        slide();
        interval = setInterval(function(){
            slide();
        }, 8000);
    }

	function makeId() {
		var text = '',
			possible = 'abcdefghijklmnopqrstuvwxyz0123456789';

		for( var i=0; i < 15; i++ )
			text += possible.charAt(Math.floor(Math.random() * possible.length));

		return text;
	}

    var playerTemplate = '<div class="mdtc-clnplrv-video mdtc-clnplrv-main-player">' +
            '<div class="mdtc-clnplrv-jplayer mdtc-clnplrv-video-container"><!-- --></div>' +
            '<div class="mdtc-clnplrv-type-playlist">' +
                '<div class="mdtc-clnplrv-gui mdtc-clnplrv-interface">' +
                    '<ul class="mdtc-clnplrv-controls">' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-previous mdtc-clnplrv-icon-prev" tabindex="1"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-play mdtc-clnplrv-icon-play" tabindex="1"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-pause mdtc-clnplrv-icon-pause" tabindex="1" style="display:none"></a></li>' +
                        '<li><a href="javascript:;" class="mdtc-clnplrv-next mdtc-clnplrv-icon-next" tabindex="1"></a></li>' +
                    '</ul>' +
                    '<div class="mdtc-clnplrv-seek-container">' +
                        '<div class="mdtc-clnplrv-seek-container-inner">' +
                            '<div class="mdtc-clnplrv-song-title">' +
                                '<div class="mdtc-clnplrv-song-title-inner">' +
                                    '<strong id="current-song"><!-- --></strong>' +
                                '</div>' +
                            '</div>' +
                            '<div class="mdtc-clnplrv-progress">' +
                                '<div class="mdtc-clnplrv-seek-bar" style="width:0%">' +
                                    '<div class="mdtc-clnplrv-play-bar" style="width:0%"><!-- --></div>' +
                                '</div>' +
                            '</div>' +
                            '<div class="mdtc-clnplrv-time">' +
                                '<div class="mdtc-clnplrv-current-time">00:00</div>' +
                                '<div class="mdtc-clnplrv-duration">00:00</div>' +
                            '</div>' +
                        '</div>' +
                    '</div>' +
					'<div class="mdtc-clnplrv-volume-bar">' +
						'<div class="mdtc-clnplrv-volume-bar-value"><i><!-- --></i></div>' +
					'</div>' +
					'<ul class="mdtc-clnplrv-toggles">' +
						'<li><a title="Repeat" tabindex="1" class="mdtc-clnplrv-repeat mdtc-clnplrv-icon-repeat" href="javascript:;"></a></li>' +
						'<li><a title="Repeat off" tabindex="1" class="mdtc-clnplrv-repeat-off mdtc-clnplrv-icon-repeat" href="javascript:;" style="display:none"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-shuffle mdtc-clnplrv-icon-shuffle" tabindex="1" title="Shuffle"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-shuffle-off mdtc-clnplrv-icon-shuffle" tabindex="1" title="Shuffle off" style="display:none"></a></li>' +
						'<li class="mdtc-clnplrv-fullscreen-li"><a href="javascript:;" class="mdtc-clnplrv-fullscreen-btn mdtc-clnplrv-icon-fullscreen" tabindex="1" title="Toggle fullscreen"></a></li>' +
						'<li><a href="javascript:;" class="mdtc-clnplrv-download-one mdtc-clnplrv-icon-download" tabindex="1" title="Download" style="display:none"></a></li>' +
					'</ul>' +
                '</div>' +
                '<div class="mdtc-clnplrv-playlist">' +
                    '<ul></ul>' +
                '</div>' +
                '<div class="mdtc-clnplrv-no-solution" style="display:none">' +
                    '<span>Update Required </span>' +
                    'To play the media you will need to either update your browser to a recent version' +
                '</div>' +
            '</div>' +
        '</div>';

    $.fn.cleanvideoplayer = function(options) {

        (function(down,  dlePlayerVideo, fn, $target){

            $('body').on('mousemove', '.mdtc-clnplrv-volume-bar', function(e){
                dlePlayerVideo && dlePlayerVideo[fn]($.extend({}, e, {currentTarget: $target}));
            }).on('mouseup', function(){
                dlePlayerVideo = false;
            }).on('mousedown', function(e){
                var $nodes = $(e.target).parents().add(e.currentTarget),
                    $volume = $nodes.filter('.mdtc-clnplrv-volume-bar').first(),
                    $seek = $nodes.filter('.mdtc-clnplrv-seek-bar').first();

                if ($seek.length) {
                    fn = 'seekBar';
                    $target = $seek;
                } else if ($volume.length) {
                    fn = 'volumeBar';
                    $target = $volume;
                }

                var $audio = $target && $target.closest('.mdtc-clnplrv-video');
                dlePlayerVideo = $audio && $audio.children().first().add($audio.prev()).filter(function(){
                        return $(this).data('dlePlayerVideo');
                    }).first().data('dlePlayerVideo');
            });

        })();

        return this.each(function(index) {

			var settings = $.extend({
				theme: 'default',
				supplied: 'm4v,webmv,youtube',
				disableControls: false,
				autoplay: false,
				playlistHeight: '290px',
				playlist: false
			}, options );

			var soloDownload = false,
				soloDownloadUrl = '';

			if ($(this).find('> ul').length > 0) {

				var ul = $(this).find('> ul');

				ul.data('theme') ? settings.theme = ul.data('theme') : void(0);
				ul.data('supplied') ? settings.supplied = ul.data('supplied') : void(0);
				ul.data('disable-controls') ? settings.disableControls = ul.data('disable-controls') : void(0);
				ul.data('autoplay') ? settings.autoplay = ul.data('autoplay') : void(0);
				ul.data('playlist-height') ? settings.playlistHeight = ul.data('playlist-height') : void(0);
				ul.data('preload') ? settings.preload = ul.data('preload') : void(0);

				if ($(this).find('> ul li').length > 0) {
					var itemsArray = [];
					$(this).find('> ul li').each(function(){
						var li		= $(this),
							free	= li.data('free') ? true : false,
							type 	= li.data('type'),
							obj		= {title: li.data('title'), artist: li.data('artist'), free: free, poster: li.data('poster')};

							obj[type] = li.data('url');

						itemsArray.push(obj);
					});

					if ($(this).find('> ul li').length == 1 && $(this).find('> ul li').data('free')) {
						soloDownload = true;
						soloDownloadUrl = $(this).find('> ul li').data('url');
					}

					settings.playlist = itemsArray;
				}
			}

			var currentId = makeId();
            $(this).html(playerTemplate);
            $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-cleanvideoplayer-id-' + currentId);

			if (soloDownload) {
				$(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-solo-download');
				$(this).find('.mdtc-clnplrv-download-one').attr('href', soloDownloadUrl);
			}

            if (settings.theme == 'dark') {
                $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-dark-theme');
            }

            if (settings.disableControls) {
                $(this).find('.mdtc-clnplrv-main-player').addClass('mdtc-clnplrv-disabled-controls');
            }

            if (settings.playlistHeight) {
                $(this).find('.mdtc-clnplrv-playlist > ul').css({
                    maxHeight: settings.playlistHeight,
                    overflowY: 'auto'
                });
            }

            if (settings.playlist.length > 1) {
                playerOptions = {
                    solution: 'html',
                    supplied: settings.supplied,
                    preload: settings.preload,
                    wmode: 'window',
                    ready: function (event) {
                        var that = $(this);

                        if (settings.autoplay) {
                            that.dlePlayerVideo('play');
                        }
						that.on('click', function() {
							that.data().dlePlayerVideo.status.paused ? that.dlePlayerVideo('play') : that.dlePlayerVideo('pause');
						});
						that.on('dblclick', function() {
							if (that.parent().hasClass('mdtc-clnplrv-state-full-screen')) {
								that.dlePlayerVideo('restoreScreen');
							} else {
								that.dlePlayerVideo('fullScreen');
							}
						});
						that.next().find('.mdtc-clnplrv-fullscreen-btn').on('click', function() {
							if (that.parent().hasClass('mdtc-clnplrv-state-full-screen')) {
								that.dlePlayerVideo('restoreScreen');
							} else {
								that.dlePlayerVideo('fullScreen');
							}
						});

						if(settings.preload == 'metadata') {
							that.find('video').css('cssText', 'background: transparent');
						}

						function resizeView() {
							if (that.next().is(':visible')) {
								var width	= that.width(),
									height	= width/16*9;
									if(that.parent().hasClass( "mdtc-clnplrv-state-full-screen" )) {
										that.css('cssText', 'height:100% !important');
									} else {
										that.css('cssText', 'height:' + height + 'px !important');
									}
								if (width < 450) {
									that.parent().addClass('mdtc-clnplrv-responsive');
								} else {
									that.parent().removeClass('mdtc-clnplrv-responsive');
								}
							}
						};
						$(window).on('resize', function() {
							resizeView();
						});
						resizeView();
						setInterval(function(){
							resizeView();
						}, 250)
						marquee(that);
                    },
                    play: function (event) {
                        $(this).dlePlayerVideo("pauseOthers");
                    },
                    cssSelector: {
                        title: '#current-song'
                    },
					playlistOptions: {
						shuffleTime: 150
					},
					size: {
						width: "100%",
						height: "100%"
					}
                };

                var playerListFiles = settings.playlist;

                var playerSelectors = {
                    dlePlayerVideo: '.mdtc-clnplrv-cleanvideoplayer-id-' + currentId + ' .mdtc-clnplrv-jplayer',
                    cssSelectorAncestor: '.mdtc-clnplrv-cleanvideoplayer-id-' + currentId
                };

                var initPlayerVideo = new dlePlayerVideoPlaylist(playerSelectors, playerListFiles, playerOptions);

            } else {
                $('.mdtc-clnplrv-cleanvideoplayer-id-' + currentId + ' .mdtc-clnplrv-jplayer').dlePlayerVideo({
                    ready: function (event) {
                        $(this).dlePlayerVideo('setMedia', settings.playlist[0]);
                        var that = $(this);

                        if (settings.autoplay) {
                            $(this).dlePlayerVideo('play');
                        }
						that.on('click', function() {
							that.data().dlePlayerVideo.status.paused ? that.dlePlayerVideo('play') : that.dlePlayerVideo('pause');
						});
						that.on('dblclick', function() {
							if (that.parent().hasClass('mdtc-clnplrv-state-full-screen')) {
								that.dlePlayerVideo('restoreScreen');
							} else {
								that.dlePlayerVideo('fullScreen');
							}
						});
						that.next().find('.mdtc-clnplrv-fullscreen-btn').on('click', function() {
							if (that.parent().hasClass('mdtc-clnplrv-state-full-screen')) {
								that.dlePlayerVideo('restoreScreen');
							} else {
								that.dlePlayerVideo('fullScreen');
							}
						});
                        $(this).parent().addClass('mdtc-clnplrv-disabled-controls mdtc-clnplrv-solo-player');
						function resizeView() {
							if (that.next().is(':visible')) {
								var width	= that.width(),
									height	= width/16*9;
									if(that.parent().hasClass( "mdtc-clnplrv-state-full-screen" )) {
										that.css('cssText', 'height:100% !important');
									} else {
										that.css('cssText', 'height:' + height + 'px !important');
									}
								if (width < 450) {
									that.parent().addClass('mdtc-clnplrv-responsive');
								} else {
									that.parent().removeClass('mdtc-clnplrv-responsive');
								}
							}
						};
						$(window).on('resize', function() {
							resizeView();
						});
						resizeView();
						setInterval(function(){
							resizeView();
						}, 250)
						marquee(that);
                    },
                    play: function (event) {
                        $(this).dlePlayerVideo('pauseOthers');
                    },
                    cssSelector: {
                        title: '#current-song'
                    },
                    cssSelectorAncestor: '.mdtc-clnplrv-cleanvideoplayer-id-' + currentId,
                    supplied: settings.supplied,
                    preload: settings.preload,
					size: {
						width: "100%",
						height: "100%"
					}
                });
            }

        });
    };




	var jPlayerYouTube = {
		api: 'https://www.youtube.com/iframe_api',

		load: (function(globals) {
			var tag = 'script',
				first = document.getElementsByTagName('head').item(0),
				queue = [],
				script,
				loading;

			return function(callback) {
				// Add callback to queue
				if (loading) {
					return (queue) ? queue.push(callback) : callback();
				}

				// Document already has youtube api
				if (globals.YT && globals.YT.Player) {
					return callback();
				}
				// Global handler already installed
				else if (globals.onYouTubeIframeAPIReady) {
					queue.push(globals.onYouTubeIframeAPIReady);
				}

				// Register new global handler.
				globals.onYouTubeIframeAPIReady = function() {
					while (queue.length !== 0) {
						try {
							queue.shift()();
						} catch (e) { }
					}
					queue = false;
				};

				loading = queue.push(callback);
				script = document.createElement(tag);
				script.src = jPlayerYouTube.api;
				first.appendChild(script);
			};
		})((0, eval)('this')),

		player: function(mediaElement, callback) {
			var $mediaElement = $(mediaElement),
				key = mediaElement.id + '_youtube',
				data = function() {
					return $mediaElement.data.apply($mediaElement, [key].concat([].slice.call(arguments)));
				},
				meta = data() || data({player: false, queue: false}) && data();

			// Player already activated
			if (meta.player) {
				return callback(meta.player);
			}
			// Player need to be activated (queue is not empty)
			else if (meta.queue) {
				return meta.queue.push(callback);
			}

			// Activation process
			meta.queue = [callback];
			jPlayerYouTube.load(function() {
				meta.container = mediaElement.parentNode.insertBefore(document.createElement('div'), mediaElement);
				meta.container.id = key;

				var os_ios = $.dlePlayerVideo.platform.ipad || $.dlePlayerVideo.platform.iphone || $.dlePlayerVideo.platform.ipod;

				if(os_ios) {var player_controls = 1;} else{var player_controls = 0;}

				meta.player = new YT.Player(key, {
					height: $mediaElement.outerHeight(),
					width: $mediaElement.outerWidth(),
					playerVars: {
						'autoplay': 0,
						'controls': player_controls,
						'rel': 0,
						'iv_load_policy': 3,
						'showinfo': 0
					},
					events: {
						onReady: function() {
							// Initialization queue
							while (meta.queue.length !== 0) {
								try {
									meta.queue.shift()(meta.player);
								} catch(e){}
							}
							delete(meta.queue);

							// Install resize handler (responsible version)
							$(window).on('resize', function(){
								meta.player.setSize(
									$mediaElement.outerWidth(),
									$mediaElement.outerHeight()
								);
							});
						},
						onStateChange: function(e) {
							var state = ['unstarted', 'ended', 'playing', 'paused', 'buffering', 'video cued'][e.data + 1],
								params,
								jPlayer = mediaElement.getJPlayer(),
								update = function (override){
									jPlayer._getHtmlStatus(params, override);
									jPlayer._updateInterface();
									if (override) {
										update();
									}
								};

							clearInterval(meta.interval);
							if (state === 'ended') {
								mediaElement.emulateEvent('ended');
							}
							else if (state === 'paused') {
								mediaElement.emulateEvent('pause');
							}
							else if (state === 'playing') {
								mediaElement.emulateEvent('play');
								mediaElement.emulateEvent('playing');

								// Emulation of time
								params = {
									currentTime: meta.player.getCurrentTime(),
									duration: meta.player.getDuration(),
									videoWidth: $mediaElement.outerWidth(),
									videoHeight: $mediaElement.outerHeight(),
									ended: false,
									readyState: 4,
									load: function() {

									},
									seekable: {
										length: 1,
										end: function() {
											return params.duration;
										}
									}
								};

								// Update state
								update(true);
								meta.interval = setInterval(function() {
									params.currentTime = meta.player.getCurrentTime();
									params.ended = (params.currentTime >= params.duration);
									update();
								}, 1e3);

								// Save original seek.
								if (!meta._html_playHead) {
									meta._html_playHead = jPlayer._html_playHead;
								}

								// Replace seek.
								jPlayer._html_playHead = function() {
									var media = this.htmlElement.media,
										currentTime = meta.player.getCurrentTime(),
										returnValue;

									if (media.getVideoId()) {
										params.currentTime = currentTime;
										this.htmlElement.media = params;
										returnValue = meta._html_playHead.apply(this, arguments);
										this.htmlElement.media = media;

										if (params.currentTime !== currentTime) {
											meta.player.seekTo(params.currentTime);
											update();
										}
									} else {
										returnValue = meta._html_playHead.apply(this, arguments);
									}

									return returnValue;
								}
							}
						}
					}
				});
			});
		},

		hooks: {
			'video': {
				getVideoId: function() {
					return function() {
						var $media = $(this);
						return (String($media.attr('src') || $media.data('src')).match(/(?:youtube\.com\/.*?[?&]v=|youtube:\/{2})([^&]+)/i) || [])[1];
					};
				},
				getJPlayer: function() {
					return function() {
						var node = this, player;
						do {
							player = $.data(node, 'dlePlayerVideo');
							node = node.parentNode;
						} while (node && !player);

						return player;
					};
				},
				emulateEvent: function() {
					return function(eventType) {
						if (typeof (this.dispatchEvent) === "function") {
							var event = document.createEvent('HTMLEvents');
							event.initEvent(eventType, true, true, window, 0, 0, 0, 0, 0, false, false, false, false, 0, this);
							this.dispatchEvent(event);
						}
						else if (typeof (this.fireEvent) !== "undefined") {
							this.fireEvent("on" + eventType);
						}
					};
				},
				youtube: function() {
					return function(callback) {
						return jPlayerYouTube.player(this, callback);
					};
				},
				toggleVideo: function() {
					return function(visible) {
						this.getJPlayer().element.toggleClass('youtube-player', visible)
							.find('video,img[id*="poster"]').css(
							visible ? {position: 'absolute', visibility: 'visible', top: '1px'}
								: {position: '', visibility: '', opacity: '', top: ''}
						)
							.end()
							.find('iframe')
							.toggle(visible);

							var os_ios = $.dlePlayerVideo.platform.ipad || $.dlePlayerVideo.platform.iphone || $.dlePlayerVideo.platform.ipod;

							if(visible && os_ios) {
								this.getJPlayer().element.parent().find('.mdtc-clnplrv-interface').toggle(false);
							} else {
								this.getJPlayer().element.parent().find('.mdtc-clnplrv-interface').toggle(true);
							}

					};
				},
				addEventListener: function(underlying) {
					return function(type, listener, useCapture) {
						var media = this,
							$media = $(this);

						return underlying.call(this, type, function(e) {
							var returnValue = listener.apply(this, arguments);

							if (media.getVideoId()) {
								if ($media.attr('src') && /(?:abort|emptied|loadstart)/i.test(type)) {
									// prevent loading of youtube link
									$media.data('src', media.src).attr('src', '');
								}
								else if (/volumechange/i.test(type)) {
									media.youtube(function(player) {
										player.setVolume(media.getJPlayer().options.volume * 100);
									});
								}
								else if (/error/i.test(type)){
									media.getJPlayer().status.waitForLoad = false;
								}
							}
							return returnValue;
						}, useCapture);
					}
				},
				'canPlayType': function(underlying) {
					return function(type) {
						return (/youtube/i.test(type))
							? 'probably'
							: underlying.apply(this, arguments);
					};
				},
				'pause': function(underlying) {
					return function() {
						var media = this;

						if (this.getVideoId()) {
							return this.youtube(function(player) {
								media.emulateEvent('pause');
								player.pauseVideo();
							});
						}

						return underlying.apply(this, arguments);
					};
				},
				'play': function(underlying) {
					return function() {
						var media = this;

						if (this.getVideoId()) {
							return this.youtube(function(player) {
								media.toggleVideo(true);
								player.playVideo();
							});
						} else if ($(media).data('src')) {
							this.youtube(function(player) {
								player.stopVideo();
							});
						}

						this.toggleVideo(false);
						return underlying.apply(this, arguments);
					};
				},
				'load': function(underlying) {
					return function() {
						var media = this,
							videoId = this.getVideoId();

						if (videoId) {
							return this.youtube(function(player) {
								media.toggleVideo(true);
								player.cueVideoById(videoId);
							});
						}

						this.toggleVideo(false);
						return underlying.apply(this, arguments);
					};
				}
			}
		},

		initialize: function() {
			var install = function(returnValue, tagName){
				var hooks = jPlayerYouTube.hooks[tagName] || {};
				for (var key in hooks){
					if (hooks.hasOwnProperty(key)) {
						returnValue[key] = hooks[key].call(returnValue, returnValue[key]);
					}
				}
				return returnValue;
			};

			$.dlePlayerVideo.prototype.format.youtube = {
				flashCanPlay: false,
				codec: 'video/youtube; codecs="m4v"',
				media: 'video'
			};

			if (typeof(HTMLVideoElement) !== 'undefined'){
				install(HTMLVideoElement.prototype, 'video');
			} else {
				document.createElement = (function(createElement) {
					return function(tagName) {
						return install(createElement.apply(this, arguments), String(tagName).toLowerCase());
					};
				})(document.createElement);
			}
		}
	};

	jPlayerYouTube.initialize();
}(jQuery));


jQuery(function($){
	$('.dleaudioplayer').cleanaudioplayer();$('.dlevideoplayer').cleanvideoplayer();
});