<?php

declare(strict_types=1);

/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2018 Spomky-Labs
 *
 * This software may be modified and distributed under the terms
 * of the MIT license.  See the LICENSE file for details.
 */

namespace Jose\Component\KeyManagement\Tests;

use Jose\Component\Core\JWK;
use Jose\Component\KeyManagement\JWKFactory;
use Jose\Component\KeyManagement\KeyConverter\KeyConverter;
use PHPUnit\Framework\TestCase;

/**
 * @see http://fm4dd.com/openssl/certexamples.htm
 *
 * @group Certificate
 * @group Unit
 */
class CertificateTest extends TestCase
{
    /**
     * @expectedException \InvalidArgumentException
     * @expectedExceptionMessage File "file:///foo/bar" does not exist.
     *
     * @test
     */
    public function fileNotFound()
    {
        KeyConverter::loadKeyFromCertificateFile('file:///foo/bar');
    }

    /**
     * @expectedException \InvalidArgumentException
     *
     * @test
     */
    public function fileNotValid()
    {
        KeyConverter::loadKeyFromCertificateFile(__DIR__.__FILE__);
    }

    /**
     * @test
     */
    public function certificateConversion()
    {
        $details = KeyConverter::loadFromKeyFile(__DIR__.'/Keys/RSA/private.encrypted.key', 'tests');
        static::assertEquals($details, [
            'kty' => 'RSA',
            'n' => 'tpS1ZmfVKVP5KofIhMBP0tSWc4qlh6fm2lrZSkuKxUjEaWjzZSzs72gEIGxraWusMdoRuV54xsWRyf5KeZT0S-I5Prle3Idi3gICiO4NwvMk6JwSBcJWwmSLFEKyUSnB2CtfiGc0_5rQCpcEt_Dn5iM-BNn7fqpoLIbks8rXKUIj8-qMVqkTXsEKeKinE23t1ykMldsNaaOH-hvGti5Jt2DMnH1JjoXdDXfxvSP_0gjUYb0ektudYFXoA6wekmQyJeImvgx4Myz1I4iHtkY_Cp7J4Mn1ejZ6HNmyvoTE_4OuY1uCeYv4UyXFc1s1uUyYtj4z57qsHGsS4dQ3A2MJsw',
            'e' => 'AQAB',
            'p' => '5BGU1c7af_5sFyfsa-onIJgo5BZu8uHvz3Uyb8OA0a-G9UPO1ShLYjX0wUfhZcFB7fwPtgmmYAN6wKGVce9eMAbX4PliPk3r-BcpZuPKkuLk_wFvgWAQ5Hqw2iEuwXLV0_e8c2gaUt_hyMC5-nFc4v0Bmv6NT6Pfry-UrK3BKWc',
            'd' => 'Kp0KuZwCZGL1BLgsVM-N0edMNitl9wN5Hf2WOYDoIqOZNAEKzdJuenIMhITJjRFUX05GVL138uyp2js_pqDdY9ipA7rAKThwGuDdNphZHech9ih3DGEPXs-YpmHqvIbCd3GoGm38MKwxYkddEpFnjo8rKna1_BpJthrFxjDRhw9DxJBycOdH2yWTyp62ZENPvneK40H2a57W4QScTgfecZqD59m2fGUaWaX5uUmIxaEmtGoJnd9RE4oywKhgN7_TK7wXRlqA4UoRPiH2ACrdU-_cLQL9Jc0u0GqZJK31LDbOeN95QgtSCc72k3Vtzy3CrVpp5TAA67s1Gj9Skn-CAQ',
            'q' => 'zPD-B-nrngwF-O99BHvb47XGKR7ON8JCI6JxavzIkusMXCB8rMyYW8zLs68L8JLAzWZ34oMq0FPUnysBxc5nTF8Nb4BZxTZ5-9cHfoKrYTI3YWsmVW2FpCJFEjMs4NXZ28PBkS9b4zjfS2KhNdkmCeOYU0tJpNfwmOTI90qeUdU',
            'dp' => 'aJrzw_kjWK9uDlTeaES2e4muv6bWbopYfrPHVWG7NPGoGdhnBnd70-jhgMEiTZSNU8VXw2u7prAR3kZ-kAp1DdwlqedYOzFsOJcPA0UZhbORyrBy30kbll_7u6CanFm6X4VyJxCpejd7jKNw6cCTFP1sfhWg5NVJ5EUTkPwE66M',
            'dq' => 'Swz1-m_vmTFN_pu1bK7vF7S5nNVrL4A0OFiEsGliCmuJWzOKdL14DiYxctvnw3H6qT2dKZZfV2tbse5N9-JecdldUjfuqAoLIe7dD7dKi42YOlTC9QXmqvTh1ohnJu8pmRFXEZQGUm_BVhoIb2_WPkjav6YSkguCUHt4HRd2YwE',
            'qi' => 'BocuCOEOq-oyLDALwzMXU8gOf3IL1Q1_BWwsdoANoh6i179psxgE4JXToWcpXZQQqub8ngwE6uR9fpd3m6N_PL4T55vbDDyjPKmrL2ttC2gOtx9KrpPh-Z7LQRo4BE48nHJJrystKHfFlaH2G7JxHNgMBYVADyttN09qEoav8Os',
        ]);

        $details = KeyConverter::loadFromKeyFile(__DIR__.'/Keys/RSA/public.key');
        static::assertEquals($details, [
            'kty' => 'RSA',
            'n' => 'tpS1ZmfVKVP5KofIhMBP0tSWc4qlh6fm2lrZSkuKxUjEaWjzZSzs72gEIGxraWusMdoRuV54xsWRyf5KeZT0S-I5Prle3Idi3gICiO4NwvMk6JwSBcJWwmSLFEKyUSnB2CtfiGc0_5rQCpcEt_Dn5iM-BNn7fqpoLIbks8rXKUIj8-qMVqkTXsEKeKinE23t1ykMldsNaaOH-hvGti5Jt2DMnH1JjoXdDXfxvSP_0gjUYb0ektudYFXoA6wekmQyJeImvgx4Myz1I4iHtkY_Cp7J4Mn1ejZ6HNmyvoTE_4OuY1uCeYv4UyXFc1s1uUyYtj4z57qsHGsS4dQ3A2MJsw',
            'e' => 'AQAB',
        ]);
    }

    /**
     * @dataProvider dataLoadCertificate
     *
     * @test
     */
    public function loadCertificate(string $file, array $expected_values)
    {
        $result = KeyConverter::loadKeyFromCertificateFile($file);

        static::assertEquals($expected_values, $result);
    }

    public function dataLoadCertificate()
    {
        return [
            [
                __DIR__.'/RSA/PEM/512b-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'm_xmkHmEQrurE_0re_jeFRLl8ZPjBop7uLHhnia7lQG_5zDtZIUC3RVpqDSwBuw_NTweGyuP-o8AG98HxqxTBw',
                    'e' => 'AQAB',
                    'x5t' => 'Bxy5TwzIUU0CQSRwjuiyaHvX2dU',
                    'x5t#256' => 'Xw-1FmWBquZKEBwVg7G-vnToFKkeeooUuh6DXXj26ec',
                    'x5c' => ['MIICEjCCAXsCAg36MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNjU0WhcNMTcwODIxMDUyNjU0WjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wXDANBgkqhkiG9w0BAQEFAANLADBIAkEAm/xmkHmEQrurE/0re/jeFRLl'.PHP_EOL.'8ZPjBop7uLHhnia7lQG/5zDtZIUC3RVpqDSwBuw/NTweGyuP+o8AG98HxqxTBwID'.PHP_EOL.'AQABMA0GCSqGSIb3DQEBBQUAA4GBABS2TLuBeTPmcaTaUW/LCB2NYOy8GMdzR1mx'.PHP_EOL.'8iBIu2H6/E2tiY3RIevV2OW61qY2/XRQg7YPxx3ffeUugX9F4J/iPnnu1zAxxyBy'.PHP_EOL.'2VguKv4SWjRFoRkIfIlHX0qVviMhSlNy2ioFLy7JcPZb+v3ftDGywUqcBiVDoea0'.PHP_EOL.'Hn+GmxZA'],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/1024b-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'xgEGvHk-U_RY0j9l3MP7o-S2a6uf4XaRBhu1ztdCHz8tMG8Kj4_qJmgsSZQD17sRctHGBTUJWp4CLtBwCf0zAGVzySwUkcHSu1_2mZ_w7Nr0TQHKeWr_j8pvXH534DKEvugr21DAHbi4c654eLUL-JW_wJJYqJh7qHM3W3Fh7ys',
                    'e' => 'AQAB',
                    'x5t' => '4bK45ewZ00Wk-a_shpTw2cCqJc8',
                    'x5t#256' => '5F5GTPOxBGAOsVyuYzqUBjri0R2YDTiDowiQbs6oGgU',
                    'x5c' => ['MIICVjCCAb8CAg37MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNzIzWhcNMTcwODIxMDUyNzIzWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAMYBBrx5PlP0WNI/ZdzD'.PHP_EOL.'+6Pktmurn+F2kQYbtc7XQh8/LTBvCo+P6iZoLEmUA9e7EXLRxgU1CVqeAi7QcAn9'.PHP_EOL.'MwBlc8ksFJHB0rtf9pmf8Oza9E0Bynlq/4/Kb1x+d+AyhL7oK9tQwB24uHOueHi1'.PHP_EOL.'C/iVv8CSWKiYe6hzN1txYe8rAgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAASPdjigJ'.PHP_EOL.'kXCqKWpnZ/Oc75EUcMi6HztaW8abUMlYXPIgkV2F7YanHOB7K4f7OOLjiz8DTPFf'.PHP_EOL.'jC9UeuErhaA/zzWi8ewMTFZW/WshOrm3fNvcMrMLKtH534JKvcdMg6qIdjTFINIr'.PHP_EOL.'evnAhf0cwULaebn+lMs8Pdl7y37+sfluVok='],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/2048b-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'tM_RXjMp7AvPrnb1_i3ImcZ4ebkY-AvUurTXngJSBgn0GJNM1HDRQqApE5JzUHf2BImsAyzW8QarrWzA2dWmq8rNWtJWJlHlSwiKr8wZDyU0kLAqKUEPVfFrk9uds8zc7OvHVRjXQiXeSTUUMpKcHsZp4zz79Jr4-4vF4Bt-_U8luj_llleaJHlJFyfXiUtqLg2HUdkjPQaFVvhYMQ7ugZl4aM1uRH7J2oxaexy_JEApSNEDnO_cripd-Pdqx-m8xbBZ9pX8FsvYnO3D_BKQk3hadbRWg_r8QYT2ZHk0NRyseoUOc3hyAeckiSWe2n9lvK-HkxmM23UVtuAwxwj4WQ',
                    'e' => 'AQAB',
                    'x5t' => 'y17eUFeZUYeOLmcTxTvpOOsjfkA',
                    'x5t#256' => 'B4plbjZwSZyZG7AnRoIFive9wF_EYsYF8PiVgXmBbNc',
                    'x5c' => ['MIIC2jCCAkMCAg38MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNzQxWhcNMTcwODIxMDUyNzQxWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC0z9FeMynsC8+u'.PHP_EOL.'dvX+LciZxnh5uRj4C9S6tNeeAlIGCfQYk0zUcNFCoCkTknNQd/YEiawDLNbxBqut'.PHP_EOL.'bMDZ1aarys1a0lYmUeVLCIqvzBkPJTSQsCopQQ9V8WuT252zzNzs68dVGNdCJd5J'.PHP_EOL.'NRQykpwexmnjPPv0mvj7i8XgG379TyW6P+WWV5okeUkXJ9eJS2ouDYdR2SM9BoVW'.PHP_EOL.'+FgxDu6BmXhozW5EfsnajFp7HL8kQClI0QOc79yuKl3492rH6bzFsFn2lfwWy9ic'.PHP_EOL.'7cP8EpCTeFp1tFaD+vxBhPZkeTQ1HKx6hQ5zeHIB5ySJJZ7af2W8r4eTGYzbdRW2'.PHP_EOL.'4DDHCPhZAgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAQMv+BFvGdMVzkQaQ3/+2noVz'.PHP_EOL.'/uAKbzpEL8xTcxYyP3lkOeh4FoxiSWqy5pGFALdPONoDuYFpLhjJSZaEwuvjI/Tr'.PHP_EOL.'rGhLV1pRG9frwDFshqD2Vaj4ENBCBh6UpeBop5+285zQ4SI7q4U9oSebUDJiuOx6'.PHP_EOL.'+tZ9KynmrbJpTSi0+BM='],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/4096b-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'sL1iEzi3sk20tUP1GnKniCyCtelBy70spiJW24k-5qQ-EjMAd-N8aSJVzeuHwtGNcpU-iy3l-ErewHCaxiFdwDJiXLA7Dc4KOe-y6rTb5zpCx9BqI4rBRCkIkRF-oDoKvbVkqsGhDXHExLAF7legENUk_hterNNIjfdoY1_Vf1eurJ0cE7Cf6eFkaS0nQI-Nu9oYjNfaiIPc64fdntq0MuxP1EoVuIKTq4YNn-n3AgZvmlyIGvqsWki3IXA1Lz166SMU3fzlkNt0IbyBM5Bmz5QQPCezcPSgsmsW2DARW9YtJQY8Ci45nKIoYiOz1bYQDrvwe9Q9oSnBYyqX7-A9VGpv9FbpisIcLoWVTYy6tQUdRSkSdQoqCxuMAk69C1YLb71MoRa0vtz3VEdE-R5QEFjzMkAx4AqWzh1tMHNIW7jXjv5UvNi44nhjRcSpjARRfZbDds7AOkMN9l5G9vxBZbVwrabjsFc7XZODA652g18vczGbqhR6b-ZVk2w1cA3chEDXJWJWwBGw3rxEKP6wDmRZfeDLut6wIC4j3mTeCHUv-PKK-SmkGgjntA7gG-BljSEONnGEOU7BB1rfhSDgDEqX_YTT4w3rtbn3-NAzrbIshnl_TVYqirbbWh6b3e629s7GrG3ABlJfnzUCY-KiJj0gfU4amaj07pBHDPzbW3k',
                    'e' => 'AQAB',
                    'x5t' => 'IBO6381r3QWOObmNaxF36HBgO5M',
                    'x5t#256' => 'yVWIatQnBpbU9lUGZmRlGg2bldGtJPpqQXfq3LhBq3M',
                    'x5c' => ['MIID2jCCA0MCAg39MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyODAwWhcNMTcwODIxMDUyODAwWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCwvWITOLeyTbS1'.PHP_EOL.'Q/UacqeILIK16UHLvSymIlbbiT7mpD4SMwB343xpIlXN64fC0Y1ylT6LLeX4St7A'.PHP_EOL.'cJrGIV3AMmJcsDsNzgo577LqtNvnOkLH0GojisFEKQiREX6gOgq9tWSqwaENccTE'.PHP_EOL.'sAXuV6AQ1ST+G16s00iN92hjX9V/V66snRwTsJ/p4WRpLSdAj4272hiM19qIg9zr'.PHP_EOL.'h92e2rQy7E/UShW4gpOrhg2f6fcCBm+aXIga+qxaSLchcDUvPXrpIxTd/OWQ23Qh'.PHP_EOL.'vIEzkGbPlBA8J7Nw9KCyaxbYMBFb1i0lBjwKLjmcoihiI7PVthAOu/B71D2hKcFj'.PHP_EOL.'Kpfv4D1Uam/0VumKwhwuhZVNjLq1BR1FKRJ1CioLG4wCTr0LVgtvvUyhFrS+3PdU'.PHP_EOL.'R0T5HlAQWPMyQDHgCpbOHW0wc0hbuNeO/lS82LjieGNFxKmMBFF9lsN2zsA6Qw32'.PHP_EOL.'Xkb2/EFltXCtpuOwVztdk4MDrnaDXy9zMZuqFHpv5lWTbDVwDdyEQNclYlbAEbDe'.PHP_EOL.'vEQo/rAOZFl94Mu63rAgLiPeZN4IdS/48or5KaQaCOe0DuAb4GWNIQ42cYQ5TsEH'.PHP_EOL.'Wt+FIOAMSpf9hNPjDeu1uff40DOtsiyGeX9NViqKtttaHpvd7rb2zsasbcAGUl+f'.PHP_EOL.'NQJj4qImPSB9ThqZqPTukEcM/NtbeQIDAQABMA0GCSqGSIb3DQEBBQUAA4GBAIAi'.PHP_EOL.'gU3My8kYYniDuKEXSJmbVB+K1upHxWDA8R6KMZGXfbe5BRd8s40cY6JBYL52Tgqd'.PHP_EOL.'l8z5Ek8dC4NNpfpcZc/teT1WqiO2wnpGHjgMDuDL1mxCZNL422jHpiPWkWp3AuDI'.PHP_EOL.'c7tL1QjbfAUHAQYwmHkWgPP+T2wAv0pOt36GgMCM'],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/8192b-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'q5lcEwG8rUflI1aL6omAaF5R1DFkCMllFaQ3HUwlwCWYNNyKxF1G2e-P3Y6SFWyp0sFfmDcvuebOY_Dw3KlC756bQUMEXH6TaubYDcXaKDyrdKgCSoufjhwHkNpRz3VxpkLADJQIHdijes2JN3daGARxSJLcjoSaZvq_LBCIHTTDGESBXJP6RtbjAjGjuLUgmcvkl029Xl8ylkrcibjTzXmOod3vioTnX5aZNT1c7evmskvixWG1NlHOhZ1HdXiPHLjKxnr4lHl9lxtTjkNSsF-Nz0bYHCpWZ9u98nkgvFAxNUmiwX5nHIqo39AK8YVuVmDGYzY-dPtD1UtCBXgj-Ryq1cPU66H7kEfvbn1kZRF0XcxqIUVDlpa_h4Aq7r8KnQ6nVF59oM8AwsrRu3llvlRzNCaRUhafZ6YUHR19j_6GpAJtOWwwV5m2qKs9EhfL9Kvz9DqWh3DBt0CuGIDS0NuBAt2_RmNQBP1u7L8pYZ_9kV-Y7YM9ocbuYpUbTy4vio33Pl2wG8iozgPIgOcbne4Vh4TGpe0hbXaL-a_84CVOWX4JyLxyBEWGB6PLfH74NyXyvh57X6Cn3P0Xr2rSgPEgEEovw5i9qDeqqzeO-GvUouhQjZgURP5USjd120IPjVoZP8RPRCAPUBQSUmi2dyHANRI3ydIhTKOEdZCVvIlVNu33wfN55kEeLCXBiDvfvozUbCGuuOkbs5Yz7vE8K9xlU_Xo2icptY_u3XMPW6YKRP6lvGtovn9295vENHOJDFCVkcJ819vHVqJnoiGAf_QX0J74NLm6fnWboH6-5BcIDl18uB3qEFAlneRflIrC2XBZju-dTuTaHy14WvVJNjTMUBgVQ4gaS1X2wmAztwv-Rk8o6k-KJuSZDWVEZyH3NaddkYSVONOMzIuuClbg4cEgLP2cxxqz8JdnyT2NNfMdGfxP4Nd_RvPtTD9kTVewlurzYVjoi8CC6VhV2Tgcp-UvT6Z0Yne-65dXi31VRqQWG8adWQ3gc9NP1oXfJqVt26ldXF9AVf7PcFcm7xzr2mwZKY-DMk1m1hBvUGeg7Iab34OABOY6J4AxXiXqKx3JV24SFydaSSevsulSrmUJU3g8TR-WwTh06Yp8DZplCU9MEvfyUSShtHED72anVRgVe8jw47k9TavJ-hPiAq0HUmmKGUeKvrqWN4bMpSiMCmHTkcqS_d4Dn4ZAI8W0DIluc9sXBaiUUSIt6t7gGNOZGUyZ-ZN4GNxVlMazB6CieGRhoPfRmXw7wq0k2R5BU1Q8PSj8jrZ88DgdfENnWCGy6Aq450OwfufGaHZDwAUD1kUoRGBkzIxvkWLVdQtmP4iZXOLSany0RtPZLGjSH-x0vQ',
                    'e' => 'AQAB',
                    'x5t' => 'YV6dSQ9sNS7rhNWcj-M4XuMmOE4',
                    'x5t#256' => 'ZNEUscWwJu03bRinDYd0BAuwiWGG3oDocehVMwX2oVo',
                    'x5c' => ['MIIF2jCCBUMCAg3+MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'OTI3MDYwNzQ5WhcNMTcwOTI2MDYwNzQ5WjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggQiMA0GCSqGSIb3DQEBAQUAA4IEDwAwggQKAoIEAQCrmVwTAbytR+Uj'.PHP_EOL.'VovqiYBoXlHUMWQIyWUVpDcdTCXAJZg03IrEXUbZ74/djpIVbKnSwV+YNy+55s5j'.PHP_EOL.'8PDcqULvnptBQwRcfpNq5tgNxdooPKt0qAJKi5+OHAeQ2lHPdXGmQsAMlAgd2KN6'.PHP_EOL.'zYk3d1oYBHFIktyOhJpm+r8sEIgdNMMYRIFck/pG1uMCMaO4tSCZy+SXTb1eXzKW'.PHP_EOL.'StyJuNPNeY6h3e+KhOdflpk1PVzt6+ayS+LFYbU2Uc6FnUd1eI8cuMrGeviUeX2X'.PHP_EOL.'G1OOQ1KwX43PRtgcKlZn273yeSC8UDE1SaLBfmcciqjf0ArxhW5WYMZjNj50+0PV'.PHP_EOL.'S0IFeCP5HKrVw9TrofuQR+9ufWRlEXRdzGohRUOWlr+HgCruvwqdDqdUXn2gzwDC'.PHP_EOL.'ytG7eWW+VHM0JpFSFp9nphQdHX2P/oakAm05bDBXmbaoqz0SF8v0q/P0OpaHcMG3'.PHP_EOL.'QK4YgNLQ24EC3b9GY1AE/W7svylhn/2RX5jtgz2hxu5ilRtPLi+Kjfc+XbAbyKjO'.PHP_EOL.'A8iA5xud7hWHhMal7SFtdov5r/zgJU5ZfgnIvHIERYYHo8t8fvg3JfK+HntfoKfc'.PHP_EOL.'/RevatKA8SAQSi/DmL2oN6qrN474a9Si6FCNmBRE/lRKN3XbQg+NWhk/xE9EIA9Q'.PHP_EOL.'FBJSaLZ3IcA1EjfJ0iFMo4R1kJW8iVU27ffB83nmQR4sJcGIO9++jNRsIa646Ruz'.PHP_EOL.'ljPu8Twr3GVT9ejaJym1j+7dcw9bpgpE/qW8a2i+f3b3m8Q0c4kMUJWRwnzX28dW'.PHP_EOL.'omeiIYB/9BfQnvg0ubp+dZugfr7kFwgOXXy4HeoQUCWd5F+UisLZcFmO751O5Nof'.PHP_EOL.'LXha9Uk2NMxQGBVDiBpLVfbCYDO3C/5GTyjqT4om5JkNZURnIfc1p12RhJU404zM'.PHP_EOL.'i64KVuDhwSAs/ZzHGrPwl2fJPY018x0Z/E/g139G8+1MP2RNV7CW6vNhWOiLwILp'.PHP_EOL.'WFXZOByn5S9PpnRid77rl1eLfVVGpBYbxp1ZDeBz00/Whd8mpW3bqV1cX0BV/s9w'.PHP_EOL.'VybvHOvabBkpj4MyTWbWEG9QZ6Dshpvfg4AE5jongDFeJeorHclXbhIXJ1pJJ6+y'.PHP_EOL.'6VKuZQlTeDxNH5bBOHTpinwNmmUJT0wS9/JRJKG0cQPvZqdVGBV7yPDjuT1Nq8n6'.PHP_EOL.'E+ICrQdSaYoZR4q+upY3hsylKIwKYdORypL93gOfhkAjxbQMiW5z2xcFqJRRIi3q'.PHP_EOL.'3uAY05kZTJn5k3gY3FWUxrMHoKJ4ZGGg99GZfDvCrSTZHkFTVDw9KPyOtnzwOB18'.PHP_EOL.'Q2dYIbLoCrjnQ7B+58ZodkPABQPWRShEYGTMjG+RYtV1C2Y/iJlc4tJqfLRG09ks'.PHP_EOL.'aNIf7HS9AgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAiXbxR0T+C6MT7Jh/SbDJ/1Gd'.PHP_EOL.'vbqskiKmmKnzOcX1x0uUHY4zHIhx3M0neYRr//XOh+FeSwM1JqAPztHy3SMRXzfP'.PHP_EOL.'xzm/nwbRwdK8C/fPy7H+uMV1mKumem8WSoOMOoxFJ+o2nJgyViwnEOu9EejlH1sc'.PHP_EOL.'uKPIoTCLUCInRRhrI84='],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/16k-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'tS3aeWW_wzlyXsDNFeBONFNq7W4lNWDjOUseNxx-R9AsqNJEWZFzaTtBI4Cam9Wf_2AlfP6i3RRpK76ooZObKwJmm1ReGcP7gf7JnODQv0W-m9x85a_fwHiI86Dhfy1YNh2zg1DO1kL_Q-sqKMOZ4g6uUfXGXjS5968sKCua3o-GEr-7GM6uw8zgpDmURtpupAFj3X1qCg6cjblPzMzcXdjACP4_zJpLc-sWpqY7pdLa26J5dgFGpTKWS7Xs96AlCPDz4uTRRFKDZarMFtzpjhWhNZyDGuYFFxNL4ca1tm-r4JyL-XuK9BTXC1WNXpqutzHNOj-tO9nCtRX02ZS3hmm1A9xndTZpfQ7lPuSA_kZEohkjcGyxtS-nup9khyMKGwvhg0MJS43VOuYSV6msk_z4dZ3-MCXVlJMTxLqWOSGHxHG0vDJQI5_IXCwkQLrVQIbt_X1ZylUdkmnKm4VuCBt4AHqK1F1jWpNXLYcFY-QW43c2Iln7v1uQFm_82CFHTanrNMBYNax2egYpSXpPS0naF6O1Y8bMPjPBU1jaoBAlfiSjCmHx5MOTg-PU9m1OnnR4XnOdDR0W8rUSS_iYz4Ucivou_7_XCTVlfuieAXT069ibXpGkTE58AgI6piVVYtaxyoADb3zr0a11Br0kS3gKRqxTq5GtgWTpz75VrFxXk8ATfwZF4PcOVX9fkUQStBKY9OGRryswLJbQ0lnz5ZR8QAAw1D2cAoWYoxUol5upjsrYrsGc7Ol3NhPPtoE0Vnxg49xQSZ0hhjhryId07aZ3QBr3NQ0XBoGhSNvO-M7ZyzDTNSUQusS5WyZsZputw_12E5y55_bbMPpKYbnx0aG93wmcna49jXoo5ZUMoJ_BQLaavGC0u-JTLT_subk5tKh3oVgDRevHwYeK1U4L4cdzpOzG8JVpcHyclofESk25DnHiQ92BfB1DR-6YadfQ8a4BXrIFlktDho1EmhXwWdT-dhO4ERPwf2Cx04iP3OdYCU_TNr3gVdB3fQLPkhV1K_od8zWrX7oqCGLkYCP_GUvl84dJoMequlyIO9IHtVpVHzGl-E48JoOHN00ULnoHzWzxUeGtda4twn9NQ-ptEjy9u0_8R-y2UqnC632wEmHpHzFqrOSYixp4GO_zAh-gmIhPJHuoH97vdcDRjGGFPO7kmMI0tBmxkt03ahYIqJKbPynHVLhsTuU7TVYrgTX6JkCR_IbudQTqVdz8oYO6tNqVrU89JI94_5ndJX1Wjmf1LPa8c31IQovBB0e-MlZ-rBkyTEttNuI8xC__OycsLhjAFx_bm0Qf2jfg2IJdLmDjGFHv3RlEdlRmJSyLY_lqKV4GAhjiEIEmduAKbygg2Jqhb6NKzHr1vxhRcWasnuhgTOunlGs3vezu9xz_4CvEKRMT6viU3tzqmGpT3zE7d0w9zMwn2eUlX0j7pKIiznrbkW2Dfe63f9X9bKYAsO5pcqcfAHqVaHl0iFXy5QoFwwjSuWwxKyhaY3tfY2rufLXCOzQ_G7BDoMRns8x6nCR-64Xuvp-EvBw0S790J_u9Z2W98rrW6c1cfn4cb9BRy3Rj64kWqlAUTRu6-qrX2RN5ywhoKfiJDH3m2q_MtgDlR3ke-5KuxaZwfM0hrcCppU5THbOwMe3XoDX-ZjD-3q-ikM8ueu4uTqDjtQrTQioFIxa-3prbNTsxBERQFZwlJtz2GmNHEAjgU-OwkMDObYAGc-ZAZritXe9vGtGFpdowMZ5k0FTUKSIsecPxn-nZlG-_qML8S63NXlU0RdbtYaLwQteFuXl_acAvuxOOnB3nZppJyIStP0uOPGhRowXSlThn0yFDht65TLly171JVrf4oFBDO4Q6EIJ7JMbRXCaEWJmeSNe_k71c3u4elbZ-C2i5JaO6bctZzO-xZ-CP7raQzHXMlpChYXqmpDU5bK2ySAbcDJDvg5WeRmQsqRxsFnI1EK1Jj_BKHZqOPz_q2SYyv69zPTsp5_w9z9YWCbOKP1KHyf9i9n6P5G3QkCzvlTDAbjR6nrrrnva0PZ0SjO4MzDOsIAa9S6vwRnWyE23vVI5RCv-IkLZ075LRkXKcj2EVPrDI3Mb1pUtfGu1H1M7m6V0SOTnaCwimIz3Ju2mwgnR-2lAAJKMd3vUaN1NfbEDuhZoMZfDrWzqOqA8Z2oyv6jHhby3DknbW4pn8tcaPCvScn1wotOeNFDvAfOIxikGEjUuXj-_gV2_dcFVIANseYpdhAS1tJjVX9JBwWcjbHnShy_9Y4f1zzrSwv4UbG7xDEGS1VaDUk5UwTTeeKQKzCkd6nYXxZYRMYDD_DcuGiCG9YvWm9hry1DkfdyCx5Pe8j8KMGUuEtIwLOIfmJDiCmE9fRAY85f9TAXyxkM-P1S_TcScKeSYrxLubX1DTuOGNDFKB4xNf0vi-lCFgLmx8tOr-RY7qtzrwrfcf7Kbpop-B5KpA2PhmoRTtZl4kF7RDeh-ZnUqcfyQcCIv_HewiMOmJ6iQDDjOWbsM8uEhl3ab-FzDYGAeT5bJs4EJAwEhsk28sXqnGzwJDUxw4mdLCYhiuI0ZwGOBUIzXQ-KHaH88PwYuQGLwM-s9uCKqJyO84I6XPe6bnqsL9NWmPhpvFxEZ7MvFCC2Z7nuGswQKpL92_QOS0NibZwzxBDUY7Qm3WsHxFzYm73JenQJGKzZPPLtjhIar7af2qb8KINgWGfIvrxR38osLT0Vg29M0DuMc',
                    'e' => 'AQAB',
                    'x5t' => 'XC_s0q4lqNalTFU6tNWR_Szk5dk',
                    'x5t#256' => '3nz2wIAoSbfVCmvy9k18bCPyIXacd3YfHrGq-qg3DVY',
                    'x5c' => ['MIIJ2jCCCUMCAg3/MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyOTAyWhcNMTcwODIxMDUyOTAyWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wgggiMA0GCSqGSIb3DQEBAQUAA4IIDwAwgggKAoIIAQC1Ldp5Zb/DOXJe'.PHP_EOL.'wM0V4E40U2rtbiU1YOM5Sx43HH5H0Cyo0kRZkXNpO0EjgJqb1Z//YCV8/qLdFGkr'.PHP_EOL.'vqihk5srAmabVF4Zw/uB/smc4NC/Rb6b3Hzlr9/AeIjzoOF/LVg2HbODUM7WQv9D'.PHP_EOL.'6yoow5niDq5R9cZeNLn3rywoK5rej4YSv7sYzq7DzOCkOZRG2m6kAWPdfWoKDpyN'.PHP_EOL.'uU/MzNxd2MAI/j/Mmktz6xampjul0trbonl2AUalMpZLtez3oCUI8PPi5NFEUoNl'.PHP_EOL.'qswW3OmOFaE1nIMa5gUXE0vhxrW2b6vgnIv5e4r0FNcLVY1emq63Mc06P6072cK1'.PHP_EOL.'FfTZlLeGabUD3Gd1Nml9DuU+5ID+RkSiGSNwbLG1L6e6n2SHIwobC+GDQwlLjdU6'.PHP_EOL.'5hJXqayT/Ph1nf4wJdWUkxPEupY5IYfEcbS8MlAjn8hcLCRAutVAhu39fVnKVR2S'.PHP_EOL.'acqbhW4IG3gAeorUXWNak1cthwVj5BbjdzYiWfu/W5AWb/zYIUdNqes0wFg1rHZ6'.PHP_EOL.'BilJek9LSdoXo7Vjxsw+M8FTWNqgECV+JKMKYfHkw5OD49T2bU6edHhec50NHRby'.PHP_EOL.'tRJL+JjPhRyK+i7/v9cJNWV+6J4BdPTr2JtekaRMTnwCAjqmJVVi1rHKgANvfOvR'.PHP_EOL.'rXUGvSRLeApGrFOrka2BZOnPvlWsXFeTwBN/BkXg9w5Vf1+RRBK0Epj04ZGvKzAs'.PHP_EOL.'ltDSWfPllHxAADDUPZwChZijFSiXm6mOytiuwZzs6Xc2E8+2gTRWfGDj3FBJnSGG'.PHP_EOL.'OGvIh3TtpndAGvc1DRcGgaFI2874ztnLMNM1JRC6xLlbJmxmm63D/XYTnLnn9tsw'.PHP_EOL.'+kphufHRob3fCZydrj2NeijllQygn8FAtpq8YLS74lMtP+y5uTm0qHehWANF68fB'.PHP_EOL.'h4rVTgvhx3Ok7MbwlWlwfJyWh8RKTbkOceJD3YF8HUNH7php19DxrgFesgWWS0OG'.PHP_EOL.'jUSaFfBZ1P52E7gRE/B/YLHTiI/c51gJT9M2veBV0Hd9As+SFXUr+h3zNatfuioI'.PHP_EOL.'YuRgI/8ZS+Xzh0mgx6q6XIg70ge1WlUfMaX4Tjwmg4c3TRQuegfNbPFR4a11ri3C'.PHP_EOL.'f01D6m0SPL27T/xH7LZSqcLrfbASYekfMWqs5JiLGngY7/MCH6CYiE8ke6gf3u91'.PHP_EOL.'wNGMYYU87uSYwjS0GbGS3TdqFgiokps/KcdUuGxO5TtNViuBNfomQJH8hu51BOpV'.PHP_EOL.'3Pyhg7q02pWtTz0kj3j/md0lfVaOZ/Us9rxzfUhCi8EHR74yVn6sGTJMS2024jzE'.PHP_EOL.'L/87JywuGMAXH9ubRB/aN+DYgl0uYOMYUe/dGUR2VGYlLItj+WopXgYCGOIQgSZ2'.PHP_EOL.'4ApvKCDYmqFvo0rMevW/GFFxZqye6GBM66eUaze97O73HP/gK8QpExPq+JTe3OqY'.PHP_EOL.'alPfMTt3TD3MzCfZ5SVfSPukoiLOetuRbYN97rd/1f1spgCw7mlypx8AepVoeXSI'.PHP_EOL.'VfLlCgXDCNK5bDErKFpje19jau58tcI7ND8bsEOgxGezzHqcJH7rhe6+n4S8HDRL'.PHP_EOL.'v3Qn+71nZb3yutbpzVx+fhxv0FHLdGPriRaqUBRNG7r6qtfZE3nLCGgp+IkMfeba'.PHP_EOL.'r8y2AOVHeR77kq7FpnB8zSGtwKmlTlMds7Ax7degNf5mMP7er6KQzy567i5OoOO1'.PHP_EOL.'CtNCKgUjFr7emts1OzEERFAVnCUm3PYaY0cQCOBT47CQwM5tgAZz5kBmuK1d728a'.PHP_EOL.'0YWl2jAxnmTQVNQpIix5w/Gf6dmUb7+owvxLrc1eVTRF1u1hovBC14W5eX9pwC+7'.PHP_EOL.'E46cHedmmknIhK0/S448aFGjBdKVOGfTIUOG3rlMuXLXvUlWt/igUEM7hDoQgnsk'.PHP_EOL.'xtFcJoRYmZ5I17+TvVze7h6Vtn4LaLklo7pty1nM77Fn4I/utpDMdcyWkKFheqak'.PHP_EOL.'NTlsrbJIBtwMkO+DlZ5GZCypHGwWcjUQrUmP8Eodmo4/P+rZJjK/r3M9Oynn/D3P'.PHP_EOL.'1hYJs4o/UofJ/2L2fo/kbdCQLO+VMMBuNHqeuuue9rQ9nRKM7gzMM6wgBr1Lq/BG'.PHP_EOL.'dbITbe9UjlEK/4iQtnTvktGRcpyPYRU+sMjcxvWlS18a7UfUzubpXRI5OdoLCKYj'.PHP_EOL.'Pcm7abCCdH7aUAAkox3e9Ro3U19sQO6Fmgxl8OtbOo6oDxnajK/qMeFvLcOSdtbi'.PHP_EOL.'mfy1xo8K9JyfXCi0540UO8B84jGKQYSNS5eP7+BXb91wVUgA2x5il2EBLW0mNVf0'.PHP_EOL.'kHBZyNsedKHL/1jh/XPOtLC/hRsbvEMQZLVVoNSTlTBNN54pArMKR3qdhfFlhExg'.PHP_EOL.'MP8Ny4aIIb1i9ab2GvLUOR93ILHk97yPwowZS4S0jAs4h+YkOIKYT19EBjzl/1MB'.PHP_EOL.'fLGQz4/VL9NxJwp5JivEu5tfUNO44Y0MUoHjE1/S+L6UIWAubHy06v5Fjuq3OvCt'.PHP_EOL.'9x/spumin4HkqkDY+GahFO1mXiQXtEN6H5mdSpx/JBwIi/8d7CIw6YnqJAMOM5Zu'.PHP_EOL.'wzy4SGXdpv4XMNgYB5PlsmzgQkDASGyTbyxeqcbPAkNTHDiZ0sJiGK4jRnAY4FQj'.PHP_EOL.'NdD4odofzw/Bi5AYvAz6z24IqonI7zgjpc97pueqwv01aY+Gm8XERnsy8UILZnue'.PHP_EOL.'4azBAqkv3b9A5LQ2JtnDPEENRjtCbdawfEXNibvcl6dAkYrNk88u2OEhqvtp/apv'.PHP_EOL.'wog2BYZ8i+vFHfyiwtPRWDb0zQO4xwIDAQABMA0GCSqGSIb3DQEBBQUAA4GBADg3'.PHP_EOL.'1Ah8wT/xjIhtUTAIcFGtgN2321aV8pIz8VSJu3CrbJJD09Ek6WUQgTbEq0pxwhQo'.PHP_EOL.'ubkr2+CJ2Gw/FTd0WFet7T57aFg7qh5xraEhH21icHmNBUG7ETUXNEf8TjbhREVY'.PHP_EOL.'gF6l8RI6rrGv0zm5awmcj+4+2OXQ+OM88dV7chMn'],
                ],
            ],
            [
                __DIR__.'/RSA/PEM/32k-rsa-example-cert.pem',
                [
                    'kty' => 'RSA',
                    'n' => 'qzPFsFIf3cSes25DloV3y3d8gKMcZVE_EQ_6e_MZnyqDbuOEP39yQs3aunzbZRoO8Xw8lLoJNduiKKsco7odI753kBvz1eLyke-sWBVZttbnYyz9AE3ZXfAb9rHW2AxgIqHNsQOsLJS_douGZwxawNdE90WM4QG80bDpkxxHfObtmZIbZoOFSeokDHA5jokQGzJ65t6ARtQOIht84pIlAr8RO0vCUiJ0R4TdAffbdIukMcVfSoZBlZJ_q-yBtPoqB1Nmr1x1FqCtR81NrEtdp7CUHy4yLIskMzHTwJL24dx8zPS9RBIAuR6HO6soQwQgKY5NYmyaZGuWDrzw0Lor9_jjcx3x7NlXEUffGyUdT_bZ6owsgd-SpvnbqXPXIf-u5JH7afSUuajytHnGVilQOpEg06B0F-AumUEx8vdLPczCx0CED11mhRhT1eRQPJlzxgqA22SN1Yz0P55R8QbfFYcflpEtZbHmdvwMSipEoEUyI8aA9z268oNVnnAGhG3cOqk8-4HOvtqZ9LIc8jUcQLtWX-PJav9EePnWuV6pFwzvKcwl09m08xIfIh9DvFVJz3Fks-X6c1tVo2Valftlj8fnlzu9WgownkwhM4KN2UpcHcff4G-v9zckhcpROSzZ1ax5mPOUMF6B2OVawMhf3li9A9JEpBDxVu2-gZU6NbhvfH1f4PdNPUnlasPylHn4qz4S6_V1fuxho-2O_V72w3V5FDBi-m2D9vDVQvJtuoiJxUEyOWaxsenuzoFlq3jNHwm0SiabwVjaMyre4qktmHopLuLX2ixME3rbTtaXLAaly-t2X6oS4nFyhwP9f_WbJb4Yh_RDxksPj1hR_4nH43NTYjZBlLDM0YRb4xRzFmATQOUhPou6LSUbl8Tl2z7WYFzlcKgHwkWRaTGUV8Sz_h-_IfgZDvCtyyLhzvWOmfJBhsV1nTbDrr8DivZGH5huBNH88v_gbCVw36aAjH8BnmcHQ0ImUUwXoiB1iWSWB3x1xdYnAyQf5RV2PK86wVc4EBRxW6MeJHWZr-kFgHtcwk2ys8MewL8xlKs1S64APAWtD-WsLGEnUVMfM5EuWjoS9kB4BI4DC6M0uyDjaCuFu80wMmWfx9C3-Y2x7l5Lw0G4gRcUk-F3ONtKfsxMqAmV6aUVkXmdkX5LLa105CpIVqflM40CPl5mlVGEFlTf9u0zclyQ0_-qWt78ZzkpolPj9XKHikdYA_DKbvtfgtgNC07GIwBctoQsOrKGOxigeWzrAwfS9S5Wt7hvcs2R0Y04rXoeSTPbHWLumsJYLxC2HPtam3IxQJzCljIOFB5Sqi9WLO5l_yjmUGS2Fzy5DkuyFuC3o79rB-Vu0zpHQ5sHdbyYkfvi3QZx4jLuj2ki-3_1Qj7RfVdd1yWeudnFUy5QGfWh3-VoaK9UIZ1EeX62owXTGNOJovn9yMdwbXmy75qrkPXadFQG3lnuqq_Ucd8ZAYJvwfQb6uhTSv1kSFCpxyyaSBYjLU44QDF6FRh_QHLMBM2DVasOT0hsF2UWsIXUneoJHk_qVZSRmj5EDaIrWAUEZfL_geiwcW3_L3Y9iaHMkB93fHNsVEpLmTO-vLHZHYN0c-kKNVBw_40xGZ5ZgPJlT4JZVvBKuB2ka2OsSLcRXZvzZZZTnrRHb_9dngGkFpI0gc6gFu2d1mPIIFp6JS7AJ4_sYKE4yxuGG7IsA4ErnNBEK9Sr1XSu0_KfcIv63dm_AybDg1vmqMLCl5EiP9OIFsWdIM42970PH9h8Ri7KUn0D53RSRVkV38NW312A2JYCHfEfbIxyibEIrsusib98x6Bedh-3BpsWyih2XlDT6AFwJdD0cc_Uf56Vqv9waUtsSx-1xBwliZ35MKq-IfV6hcLnFgLhxsqakV8aFLAEzI8Ulned6zjRAC28aaDOZcFdKEMD0wHPUW8-9UTQxAgug8otEITWSkKubyXbdofpVa9Xwjq1-jLb4eylqey0RokKrHO6B7F3KtUF8Zsm0mGEg7nvUhjEBFL3AqkLke5Nb_78uqb3tzZF3iO6ghENar9s1DUIYqNkbMSeh7smgER_PBUB0MGMqRnx8qcr5t5yBEurZ7qq7-LYoJOoc6UwaPrQN_AFRou4ugiRrxIrvOwrDPr4y2zoi9XKnBBuYMnt2AkGVCNIA0WOKgmex4x_2Nri2JlRieAPwNPfW5PLkyPVRfw0dNzhg7csMl1Wctdw1JpHJhgMswuhYhRWGyzYWE4ZU8lvQWqA42MOKfUixAV4LmEzGz2eRQSPGWjLC85-mcxf_vssmD-mbuJAjzlLDzzwllrTDCQrt18DftpAAHhD5hG2HmQH9RDzcS3sniIx4p2zyqBHVQsWM74BlQjbODjgHRHerTgxYaNmh4KRA38lmb9omrUhI2Q0Lj5CF2of_Apd7fo8u6LpBpdEtirkn_7-9vPPiGerClV6lSjoNi_I_hHCneAq-3KZq7hM5XliJPvUrws_m0X5n6_fazdk-gOohEuF0Aq_1I5633sS-DGrFyan2K7oeoBGQN994-kweTR0lLko14nC5wnvizbsv7sDUNJTjM7LMYIrhKEILTjjGQ6WuCkYhQuM4RAnx74jFIchW8pS1tEnUcIOyBWgFB9M2zdbNmJg7vH43mmX408jMYVKs9CQz2Y7Vu33S0dSp9sWxM1KUREFVy1xTbVgKNxLxOzXiLOjm_b4EifAHZh_KTf0POm5RESU-TSrO29y5puTHL-PLuOE30jrxXaKhW5UzmQLUMhBGI7geYP6fE6QxyUi0gD_tLdMmzxTlZiOXkE6HnBQ-3Ar54uA-RFUhnzU-XT3wm--eINsvqyrHCyLQlmM71aBXnMlH5g0NJjdm42XSecTopWfFCfcNe1-ufpUuMGGg0C3LxVN5fkTmB2_6gai0AHh4dNhefGkKCZ5OcSNtA_UUI1nKr_wgPTI4X1catN9RE9mMYhOt-I5gOVRCihxDcUcBl2apUaFK-jHPs5rABqhykbi_dOS-zy42I86Vcu4B-_0GNlRIPRLZWFIhNRy_kfCOq4kb4SK9DjTvHsaq6YWMoL9Jk3JiqvH4yrMZ6T-XEFdJ8DGSc41lo1YJwhFUu0eGbGFKxyUBrHv1l9ByPrqWaiepnBBsda4y8G3SoiCfndwkbvLeE5ykYgurPpkYX_bau2PqsoAkiJ_GmbitKpXD71C5PmzvzLvpxkgC6hQq-v4L4WLelADvBpeikX9k23qhR5H3mkzNeMZgHyoFisy161cDgOlcg64g6C2UzJKlb5C1tOlQwM3fdm7cjBJXOjuxgi8Ewx6ov90eeaqIEfFvnUu1_IC_tFve9P_Us21Ak53vwStlHueYHtedJsHg84C5Ppt_z1LFR3Hh8m1pOnlb3kJw5eGpvsXweZrIIN0cvwz-NZ_orIxjPxLf23wy-y-lhObK17BfX1g-p759XtRSaG4Rj_QedauXHAA-SKgvwAOY3kBuWo9Oxx73JbC1kov55TkecHj2lXO_o49O5LCOa_h0nHIVb3JIGWot11sF_6zwNzFM2WtHFNu7Iu9hllumC8rvz3HEbylvSPQYzBQKy8NSyC6T9wbH6cAYY-vl59q1J4DwBH3DHKoMAec8InlnBO_ekJa8SMdQMZxov0BaxJc0W__29w2Sza0cBsMslfpRIWRWMb4jNpyvCyEVxrGf7AakOl0_9P3JCQ2o8cuf-BGg_z_iQ3aTMYVWi_pWuxnhh5NchjQU8C3dxvnEd0Te9mmDlvZh-N9GULo0tlzHz3WZniUp7mxVQ3nkeS31M0LIIF3SetSMjXrGJ_4bzAnb3EjH44eFuvgOiJ8ChXLCmHLtIpFa0WSC6YVpBxqfPrxke-DyB2Lvz_46MSQ4iKvCFhdYWxBtwXCZDN5Dt4XFpMknL_VnuVU8a5_rRqpEebv_VF1pBZsvfTK6UXFWAApFvL4ebApuLsFInG3uk89N2SbenTTiBGWZWZjsEFsvf3iSFZdQ2bgKSLmJIsuXV1mUPkzGEr8SsPLDKhGNZBevtka-CfnukEPn7a3K_O5sYcccEtYwx0VNiC6dWu7B_-pflffa1m4pbhdg6KfykDO9_jU_LE692dhWUzbv977zGUlOnmsEMeqmSTo9V5Hv0UsEDGEjoe9piKidoZ8JdAq1WIpSBfW9M2wtkZHbi2nlaBnKJuTaaNs_nWjbG4y73hEqEqRlQMKrLsJU7rsmy3h6x6-J_tXfkKpWu_Z_PhR-ca2RV4ldwUNejBhBomg-6bcSq1lHXGTpwc0wSDmIUfE2W6ZZysaFpmGpTDFjTDqfeeAwwbzShK7Uc-OnJVNiQ5w1KALJNjXURSfI61vyWRBMtFHaC7t6ixwDfv6pqEa0xeDe4xf4Z1qdX1Zfs4xpdAyzZWmslUsXIYDtiTXq6NYGjnCEPYqneVGOWhP6re0UfzeqqB6p6_L42UoqFrrjU7jnEWRlz6gxdU9qOJgLX3u6CIYtN6b44tpsqA23fNBiuf4SqoYimbd2YVjXFRFFNZ2XqJ-wBqYcD5xIfudMN6W5cAD4p5cTQ11_-EqIp8rDxiWOs-PN8SQTIE7ZYQ6na-lSITpchNybreE9SqhzluoY71DN8oQuUJHonrAW5Hh_VroGBxpbO9XdNhw0XrC-S9iH9DDEUedanM2DznPUZsHHutG8H0K9AEyWRS01sAwrF73ZG57qy5IciYMHZuFbkY0lzwbF-vd15jgNfP4JTmZD2sVWwVgI7Qp9T2hd0uuZL_huHl2baRCyC_DSI9c6p3q9Ud_tBN_yCcNcUVx0rS6EGfzM8VYOGwyiBVBAgVDjBXiKBsUVWA3ljfOtYhLKBDHkqhvoQaczSI2fKX7L7cwgXeBdckoaNhno6mCpZBamuyBZ1Iy6TnguQi59MCCKdiczIpfeumbSDEovy2IbQmPqld_JI6WOufgldiITu3hXR5KNazan2mc3NrKu1SEXZpdzb4wJZZ26U_1xE2GLMJru05yZoVNEkN72DhagM1R5oqHwPzRcn3ahdYvUzDoP6UHEpa76A23lqafY7F98l66hmAnXXlEKzEVwthYoxWANYtVsxs9NktNJdNMB3OCMnCo9BWkefmjlrzMJSkBP_1mfxN2o3W1tMNXpk5OQPO20_eWPF3iYhobSo8fcxzXtw9bg1BXr0TADj0hl_z4jw93wVGGLlsA3qYstay0I9yJgHBZmhxc7V1JzNWdwxIDmRgA5eCm1ELVBxpIup9WGZlUs1rzwqXzI-37i7l3dwFfCf_i2g8m-gNQjuM6YqkSz-XKcn-sJEg1XSMhoB15sgYE9U-2Oe-_EGLK0dOU2zyHO40F8ghvhKWpuAcITX_QnEMremwsiCl0PEnGZ98BXzlRvd1MFNc0ZUwzN-wTVxs4jNkteNbp0MjIKA5Y6FiCEX6koNWY9cLXSNg4XG4IsWRQrfIn2WWFz_nhzlaZNm_NUM1kmKRREPmsvQ',
                    'e' => 'AQAB',
                    'x5t' => 'KGApLybHWJmBwZGgBk07AlRD9nU',
                    'x5t#256' => 'YD12k6kc4xuh_5vEHMyyOFpGs6VqTyaKMlxg0Nt2crA',
                    'x5c' => ['MIIR2jCCEUMCAg4EMA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIx'.PHP_EOL.'MDEwMTIxNzQ5WhcNMTcxMDA5MTIxNzQ5WjBKMQswCQYDVQQGDAJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wghAiMA0GCSqGSIb3DQEBAQUAA4IQDwAwghAKAoIQAQCrM8WwUh/dxJ6z'.PHP_EOL.'bkOWhXfLd3yAoxxlUT8RD/p78xmfKoNu44Q/f3JCzdq6fNtlGg7xfDyUugk126Io'.PHP_EOL.'qxyjuh0jvneQG/PV4vKR76xYFVm21udjLP0ATdld8Bv2sdbYDGAioc2xA6wslL92'.PHP_EOL.'i4ZnDFrA10T3RYzhAbzRsOmTHEd85u2Zkhtmg4VJ6iQMcDmOiRAbMnrm3oBG1A4i'.PHP_EOL.'G3zikiUCvxE7S8JSInRHhN0B99t0i6QxxV9KhkGVkn+r7IG0+ioHU2avXHUWoK1H'.PHP_EOL.'zU2sS12nsJQfLjIsiyQzMdPAkvbh3HzM9L1EEgC5Hoc7qyhDBCApjk1ibJpka5YO'.PHP_EOL.'vPDQuiv3+ONzHfHs2VcRR98bJR1P9tnqjCyB35Km+dupc9ch/67kkftp9JS5qPK0'.PHP_EOL.'ecZWKVA6kSDToHQX4C6ZQTHy90s9zMLHQIQPXWaFGFPV5FA8mXPGCoDbZI3VjPQ/'.PHP_EOL.'nlHxBt8Vhx+WkS1lseZ2/AxKKkSgRTIjxoD3Pbryg1WecAaEbdw6qTz7gc6+2pn0'.PHP_EOL.'shzyNRxAu1Zf48lq/0R4+da5XqkXDO8pzCXT2bTzEh8iH0O8VUnPcWSz5fpzW1Wj'.PHP_EOL.'ZVqV+2WPx+eXO71aCjCeTCEzgo3ZSlwdx9/gb6/3NySFylE5LNnVrHmY85QwXoHY'.PHP_EOL.'5VrAyF/eWL0D0kSkEPFW7b6BlTo1uG98fV/g9009SeVqw/KUefirPhLr9XV+7GGj'.PHP_EOL.'7Y79XvbDdXkUMGL6bYP28NVC8m26iInFQTI5ZrGx6e7OgWWreM0fCbRKJpvBWNoz'.PHP_EOL.'Kt7iqS2Yeiku4tfaLEwTettO1pcsBqXL63ZfqhLicXKHA/1/9ZslvhiH9EPGSw+P'.PHP_EOL.'WFH/icfjc1NiNkGUsMzRhFvjFHMWYBNA5SE+i7otJRuXxOXbPtZgXOVwqAfCRZFp'.PHP_EOL.'MZRXxLP+H78h+BkO8K3LIuHO9Y6Z8kGGxXWdNsOuvwOK9kYfmG4E0fzy/+BsJXDf'.PHP_EOL.'poCMfwGeZwdDQiZRTBeiIHWJZJYHfHXF1icDJB/lFXY8rzrBVzgQFHFbox4kdZmv'.PHP_EOL.'6QWAe1zCTbKzwx7AvzGUqzVLrgA8Ba0P5awsYSdRUx8zkS5aOhL2QHgEjgMLozS7'.PHP_EOL.'IONoK4W7zTAyZZ/H0Lf5jbHuXkvDQbiBFxST4Xc420p+zEyoCZXppRWReZ2Rfkst'.PHP_EOL.'rXTkKkhWp+UzjQI+XmaVUYQWVN/27TNyXJDT/6pa3vxnOSmiU+P1coeKR1gD8Mpu'.PHP_EOL.'+1+C2A0LTsYjAFy2hCw6soY7GKB5bOsDB9L1Lla3uG9yzZHRjTiteh5JM9sdYu6a'.PHP_EOL.'wlgvELYc+1qbcjFAnMKWMg4UHlKqL1Ys7mX/KOZQZLYXPLkOS7IW4Lejv2sH5W7T'.PHP_EOL.'OkdDmwd1vJiR++LdBnHiMu6PaSL7f/VCPtF9V13XJZ652cVTLlAZ9aHf5Whor1Qh'.PHP_EOL.'nUR5frajBdMY04mi+f3Ix3BtebLvmquQ9dp0VAbeWe6qr9Rx3xkBgm/B9Bvq6FNK'.PHP_EOL.'/WRIUKnHLJpIFiMtTjhAMXoVGH9AcswEzYNVqw5PSGwXZRawhdSd6gkeT+pVlJGa'.PHP_EOL.'PkQNoitYBQRl8v+B6LBxbf8vdj2JocyQH3d8c2xUSkuZM768sdkdg3Rz6Qo1UHD/'.PHP_EOL.'jTEZnlmA8mVPgllW8Eq4HaRrY6xItxFdm/NlllOetEdv/12eAaQWkjSBzqAW7Z3W'.PHP_EOL.'Y8ggWnolLsAnj+xgoTjLG4YbsiwDgSuc0EQr1KvVdK7T8p9wi/rd2b8DJsODW+ao'.PHP_EOL.'wsKXkSI/04gWxZ0gzjb3vQ8f2HxGLspSfQPndFJFWRXfw1bfXYDYlgId8R9sjHKJ'.PHP_EOL.'sQiuy6yJv3zHoF52H7cGmxbKKHZeUNPoAXAl0PRxz9R/npWq/3BpS2xLH7XEHCWJ'.PHP_EOL.'nfkwqr4h9XqFwucWAuHGypqRXxoUsATMjxSWd53rONEALbxpoM5lwV0oQwPTAc9R'.PHP_EOL.'bz71RNDECC6Dyi0QhNZKQq5vJdt2h+lVr1fCOrX6Mtvh7KWp7LRGiQqsc7oHsXcq'.PHP_EOL.'1QXxmybSYYSDue9SGMQEUvcCqQuR7k1v/vy6pve3NkXeI7qCEQ1qv2zUNQhio2Rs'.PHP_EOL.'xJ6HuyaARH88FQHQwYypGfHypyvm3nIES6tnuqrv4tigk6hzpTBo+tA38AVGi7i6'.PHP_EOL.'CJGvEiu87CsM+vjLbOiL1cqcEG5gye3YCQZUI0gDRY4qCZ7HjH/Y2uLYmVGJ4A/A'.PHP_EOL.'099bk8uTI9VF/DR03OGDtywyXVZy13DUmkcmGAyzC6FiFFYbLNhYThlTyW9BaoDj'.PHP_EOL.'Yw4p9SLEBXguYTMbPZ5FBI8ZaMsLzn6ZzF/++yyYP6Zu4kCPOUsPPPCWWtMMJCu3'.PHP_EOL.'XwN+2kAAeEPmEbYeZAf1EPNxLeyeIjHinbPKoEdVCxYzvgGVCNs4OOAdEd6tODFh'.PHP_EOL.'o2aHgpEDfyWZv2iatSEjZDQuPkIXah/8Cl3t+jy7oukGl0S2KuSf/v7288+IZ6sK'.PHP_EOL.'VXqVKOg2L8j+EcKd4Cr7cpmruEzleWIk+9SvCz+bRfmfr99rN2T6A6iES4XQCr/U'.PHP_EOL.'jnrfexL4MasXJqfYruh6gEZA333j6TB5NHSUuSjXicLnCe+LNuy/uwNQ0lOMzssx'.PHP_EOL.'giuEoQgtOOMZDpa4KRiFC4zhECfHviMUhyFbylLW0SdRwg7IFaAUH0zbN1s2YmDu'.PHP_EOL.'8fjeaZfjTyMxhUqz0JDPZjtW7fdLR1Kn2xbEzUpREQVXLXFNtWAo3EvE7NeIs6Ob'.PHP_EOL.'9vgSJ8AdmH8pN/Q86blERJT5NKs7b3Lmm5Mcv48u44TfSOvFdoqFblTOZAtQyEEY'.PHP_EOL.'juB5g/p8TpDHJSLSAP+0t0ybPFOVmI5eQToecFD7cCvni4D5EVSGfNT5dPfCb754'.PHP_EOL.'g2y+rKscLItCWYzvVoFecyUfmDQ0mN2bjZdJ5xOilZ8UJ9w17X65+lS4wYaDQLcv'.PHP_EOL.'FU3l+ROYHb/qBqLQAeHh02F58aQoJnk5xI20D9RQjWcqv/CA9MjhfVxq031ET2Yx'.PHP_EOL.'iE634jmA5VEKKHENxRwGXZqlRoUr6Mc+zmsAGqHKRuL905L7PLjYjzpVy7gH7/QY'.PHP_EOL.'2VEg9EtlYUiE1HL+R8I6riRvhIr0ONO8exqrphYygv0mTcmKq8fjKsxnpP5cQV0n'.PHP_EOL.'wMZJzjWWjVgnCEVS7R4ZsYUrHJQGse/WX0HI+upZqJ6mcEGx1rjLwbdKiIJ+d3CR'.PHP_EOL.'u8t4TnKRiC6s+mRhf9tq7Y+qygCSIn8aZuK0qlcPvULk+bO/Mu+nGSALqFCr6/gv'.PHP_EOL.'hYt6UAO8Gl6KRf2TbeqFHkfeaTM14xmAfKgWKzLXrVwOA6VyDriDoLZTMkqVvkLW'.PHP_EOL.'06VDAzd92btyMElc6O7GCLwTDHqi/3R55qogR8W+dS7X8gL+0W970/9SzbUCTne/'.PHP_EOL.'BK2Ue55ge150mweDzgLk+m3/PUsVHceHybWk6eVveQnDl4am+xfB5msgg3Ry/DP4'.PHP_EOL.'1n+isjGM/Et/bfDL7L6WE5srXsF9fWD6nvn1e1FJobhGP9B51q5ccAD5IqC/AA5j'.PHP_EOL.'eQG5aj07HHvclsLWSi/nlOR5wePaVc7+jj07ksI5r+HScchVvckgZai3XWwX/rPA'.PHP_EOL.'3MUzZa0cU27si72GWW6YLyu/PccRvKW9I9BjMFArLw1LILpP3BsfpwBhj6+Xn2rU'.PHP_EOL.'ngPAEfcMcqgwB5zwieWcE796QlrxIx1AxnGi/QFrElzRb//b3DZLNrRwGwyyV+lE'.PHP_EOL.'hZFYxviM2nK8LIRXGsZ/sBqQ6XT/0/ckJDajxy5/4EaD/P+JDdpMxhVaL+la7GeG'.PHP_EOL.'Hk1yGNBTwLd3G+cR3RN72aYOW9mH430ZQujS2XMfPdZmeJSnubFVDeeR5LfUzQsg'.PHP_EOL.'gXdJ61IyNesYn/hvMCdvcSMfjh4W6+A6InwKFcsKYcu0ikVrRZILphWkHGp8+vGR'.PHP_EOL.'74PIHYu/P/joxJDiIq8IWF1hbEG3BcJkM3kO3hcWkyScv9We5VTxrn+tGqkR5u/9'.PHP_EOL.'UXWkFmy99MrpRcVYACkW8vh5sCm4uwUicbe6Tz03ZJt6dNOIEZZlZmOwQWy9/eJI'.PHP_EOL.'Vl1DZuApIuYkiy5dXWZQ+TMYSvxKw8sMqEY1kF6+2Rr4J+e6QQ+ftrcr87mxhxxw'.PHP_EOL.'S1jDHRU2ILp1a7sH/6l+V99rWbiluF2Dop/KQM73+NT8sTr3Z2FZTNu/3vvMZSU6'.PHP_EOL.'eawQx6qZJOj1Xke/RSwQMYSOh72mIqJ2hnwl0CrVYilIF9b0zbC2RkduLaeVoGco'.PHP_EOL.'m5Npo2z+daNsbjLveESoSpGVAwqsuwlTuuybLeHrHr4n+1d+Qqla79n8+FH5xrZF'.PHP_EOL.'XiV3BQ16MGEGiaD7ptxKrWUdcZOnBzTBIOYhR8TZbplnKxoWmYalMMWNMOp954DD'.PHP_EOL.'BvNKErtRz46clU2JDnDUoAsk2NdRFJ8jrW/JZEEy0UdoLu3qLHAN+/qmoRrTF4N7'.PHP_EOL.'jF/hnWp1fVl+zjGl0DLNlaayVSxchgO2JNero1gaOcIQ9iqd5UY5aE/qt7RR/N6q'.PHP_EOL.'oHqnr8vjZSioWuuNTuOcRZGXPqDF1T2o4mAtfe7oIhi03pvji2myoDbd80GK5/hK'.PHP_EOL.'qhiKZt3ZhWNcVEUU1nZeon7AGphwPnEh+50w3pblwAPinlxNDXX/4SoinysPGJY6'.PHP_EOL.'z483xJBMgTtlhDqdr6VIhOlyE3Jut4T1KqHOW6hjvUM3yhC5QkeiesBbkeH9WugY'.PHP_EOL.'HGls71d02HDResL5L2If0MMRR51qczYPOc9Rmwce60bwfQr0ATJZFLTWwDCsXvdk'.PHP_EOL.'bnurLkhyJgwdm4VuRjSXPBsX693XmOA18/glOZkPaxVbBWAjtCn1PaF3S65kv+G4'.PHP_EOL.'eXZtpELIL8NIj1zqner1R3+0E3/IJw1xRXHStLoQZ/MzxVg4bDKIFUECBUOMFeIo'.PHP_EOL.'GxRVYDeWN861iEsoEMeSqG+hBpzNIjZ8pfsvtzCBd4F1ySho2GejqYKlkFqa7IFn'.PHP_EOL.'UjLpOeC5CLn0wIIp2JzMil966ZtIMSi/LYhtCY+qV38kjpY65+CV2IhO7eFdHko1'.PHP_EOL.'rNqfaZzc2sq7VIRdml3NvjAllnbpT/XETYYswmu7TnJmhU0SQ3vYOFqAzVHmiofA'.PHP_EOL.'/NFyfdqF1i9TMOg/pQcSlrvoDbeWpp9jsX3yXrqGYCddeUQrMRXC2FijFYA1i1Wz'.PHP_EOL.'Gz02S00l00wHc4IycKj0FaR5+aOWvMwlKQE//WZ/E3ajdbW0w1emTk5A87bT95Y8'.PHP_EOL.'XeJiGhtKjx9zHNe3D1uDUFevRMAOPSGX/PiPD3fBUYYuWwDepiy1rLQj3ImAcFma'.PHP_EOL.'HFztXUnM1Z3DEgOZGADl4KbUQtUHGki6n1YZmVSzWvPCpfMj7fuLuXd3AV8J/+La'.PHP_EOL.'Dyb6A1CO4zpiqRLP5cpyf6wkSDVdIyGgHXmyBgT1T7Y5778QYsrR05TbPIc7jQXy'.PHP_EOL.'CG+Epam4BwhNf9CcQyt6bCyIKXQ8ScZn3wFfOVG93UwU1zRlTDM37BNXGziM2S14'.PHP_EOL.'1unQyMgoDljoWIIRfqSg1Zj1wtdI2DhcbgixZFCt8ifZZYXP+eHOVpk2b81QzWSY'.PHP_EOL.'pFEQ+ay9AgMBAAEwDQYJKoZIhvcNAQEFBQADgYEATPINk/17H+RLz459iCVQAGes'.PHP_EOL.'8kc5sxYj3CkMlWrGMiCxvsgu2kak6dCa0f3DfiVt54Fry7s0OklHiZmipoiF4RCt'.PHP_EOL.'yJwUSAzRrZFAbkpDg8oIu4Ui/Bt13kY7xON+u4m0IgkLZSE+8BSjMrfjVvVxe+qH'.PHP_EOL.'5i7X/ibUTDjgyfdA8XI='],
                ],
            ],
            [
                __DIR__.'/RSA/DER/512b-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'm_xmkHmEQrurE_0re_jeFRLl8ZPjBop7uLHhnia7lQG_5zDtZIUC3RVpqDSwBuw_NTweGyuP-o8AG98HxqxTBw',
                    'e' => 'AQAB',
                    'x5t' => 'Bxy5TwzIUU0CQSRwjuiyaHvX2dU',
                    'x5t#256' => 'Xw-1FmWBquZKEBwVg7G-vnToFKkeeooUuh6DXXj26ec',
                    'x5c' => ['MIICEjCCAXsCAg36MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNjU0WhcNMTcwODIxMDUyNjU0WjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wXDANBgkqhkiG9w0BAQEFAANLADBIAkEAm/xmkHmEQrurE/0re/jeFRLl'.PHP_EOL.'8ZPjBop7uLHhnia7lQG/5zDtZIUC3RVpqDSwBuw/NTweGyuP+o8AG98HxqxTBwID'.PHP_EOL.'AQABMA0GCSqGSIb3DQEBBQUAA4GBABS2TLuBeTPmcaTaUW/LCB2NYOy8GMdzR1mx'.PHP_EOL.'8iBIu2H6/E2tiY3RIevV2OW61qY2/XRQg7YPxx3ffeUugX9F4J/iPnnu1zAxxyBy'.PHP_EOL.'2VguKv4SWjRFoRkIfIlHX0qVviMhSlNy2ioFLy7JcPZb+v3ftDGywUqcBiVDoea0'.PHP_EOL.'Hn+GmxZA'],
                ],
            ],
            [
                __DIR__.'/RSA/DER/1024b-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'xgEGvHk-U_RY0j9l3MP7o-S2a6uf4XaRBhu1ztdCHz8tMG8Kj4_qJmgsSZQD17sRctHGBTUJWp4CLtBwCf0zAGVzySwUkcHSu1_2mZ_w7Nr0TQHKeWr_j8pvXH534DKEvugr21DAHbi4c654eLUL-JW_wJJYqJh7qHM3W3Fh7ys',
                    'e' => 'AQAB',
                    'x5t' => '4bK45ewZ00Wk-a_shpTw2cCqJc8',
                    'x5t#256' => '5F5GTPOxBGAOsVyuYzqUBjri0R2YDTiDowiQbs6oGgU',
                    'x5c' => ['MIICVjCCAb8CAg37MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNzIzWhcNMTcwODIxMDUyNzIzWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wgZ8wDQYJKoZIhvcNAQEBBQADgY0AMIGJAoGBAMYBBrx5PlP0WNI/ZdzD'.PHP_EOL.'+6Pktmurn+F2kQYbtc7XQh8/LTBvCo+P6iZoLEmUA9e7EXLRxgU1CVqeAi7QcAn9'.PHP_EOL.'MwBlc8ksFJHB0rtf9pmf8Oza9E0Bynlq/4/Kb1x+d+AyhL7oK9tQwB24uHOueHi1'.PHP_EOL.'C/iVv8CSWKiYe6hzN1txYe8rAgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAASPdjigJ'.PHP_EOL.'kXCqKWpnZ/Oc75EUcMi6HztaW8abUMlYXPIgkV2F7YanHOB7K4f7OOLjiz8DTPFf'.PHP_EOL.'jC9UeuErhaA/zzWi8ewMTFZW/WshOrm3fNvcMrMLKtH534JKvcdMg6qIdjTFINIr'.PHP_EOL.'evnAhf0cwULaebn+lMs8Pdl7y37+sfluVok='],
                ],
            ],
            [
                __DIR__.'/RSA/DER/2048b-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'tM_RXjMp7AvPrnb1_i3ImcZ4ebkY-AvUurTXngJSBgn0GJNM1HDRQqApE5JzUHf2BImsAyzW8QarrWzA2dWmq8rNWtJWJlHlSwiKr8wZDyU0kLAqKUEPVfFrk9uds8zc7OvHVRjXQiXeSTUUMpKcHsZp4zz79Jr4-4vF4Bt-_U8luj_llleaJHlJFyfXiUtqLg2HUdkjPQaFVvhYMQ7ugZl4aM1uRH7J2oxaexy_JEApSNEDnO_cripd-Pdqx-m8xbBZ9pX8FsvYnO3D_BKQk3hadbRWg_r8QYT2ZHk0NRyseoUOc3hyAeckiSWe2n9lvK-HkxmM23UVtuAwxwj4WQ',
                    'e' => 'AQAB',
                    'x5t' => 'y17eUFeZUYeOLmcTxTvpOOsjfkA',
                    'x5t#256' => 'B4plbjZwSZyZG7AnRoIFive9wF_EYsYF8PiVgXmBbNc',
                    'x5c' => ['MIIC2jCCAkMCAg38MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyNzQxWhcNMTcwODIxMDUyNzQxWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC0z9FeMynsC8+u'.PHP_EOL.'dvX+LciZxnh5uRj4C9S6tNeeAlIGCfQYk0zUcNFCoCkTknNQd/YEiawDLNbxBqut'.PHP_EOL.'bMDZ1aarys1a0lYmUeVLCIqvzBkPJTSQsCopQQ9V8WuT252zzNzs68dVGNdCJd5J'.PHP_EOL.'NRQykpwexmnjPPv0mvj7i8XgG379TyW6P+WWV5okeUkXJ9eJS2ouDYdR2SM9BoVW'.PHP_EOL.'+FgxDu6BmXhozW5EfsnajFp7HL8kQClI0QOc79yuKl3492rH6bzFsFn2lfwWy9ic'.PHP_EOL.'7cP8EpCTeFp1tFaD+vxBhPZkeTQ1HKx6hQ5zeHIB5ySJJZ7af2W8r4eTGYzbdRW2'.PHP_EOL.'4DDHCPhZAgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAQMv+BFvGdMVzkQaQ3/+2noVz'.PHP_EOL.'/uAKbzpEL8xTcxYyP3lkOeh4FoxiSWqy5pGFALdPONoDuYFpLhjJSZaEwuvjI/Tr'.PHP_EOL.'rGhLV1pRG9frwDFshqD2Vaj4ENBCBh6UpeBop5+285zQ4SI7q4U9oSebUDJiuOx6'.PHP_EOL.'+tZ9KynmrbJpTSi0+BM='],
                ],
            ],
            [
                __DIR__.'/RSA/DER/4096b-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'sL1iEzi3sk20tUP1GnKniCyCtelBy70spiJW24k-5qQ-EjMAd-N8aSJVzeuHwtGNcpU-iy3l-ErewHCaxiFdwDJiXLA7Dc4KOe-y6rTb5zpCx9BqI4rBRCkIkRF-oDoKvbVkqsGhDXHExLAF7legENUk_hterNNIjfdoY1_Vf1eurJ0cE7Cf6eFkaS0nQI-Nu9oYjNfaiIPc64fdntq0MuxP1EoVuIKTq4YNn-n3AgZvmlyIGvqsWki3IXA1Lz166SMU3fzlkNt0IbyBM5Bmz5QQPCezcPSgsmsW2DARW9YtJQY8Ci45nKIoYiOz1bYQDrvwe9Q9oSnBYyqX7-A9VGpv9FbpisIcLoWVTYy6tQUdRSkSdQoqCxuMAk69C1YLb71MoRa0vtz3VEdE-R5QEFjzMkAx4AqWzh1tMHNIW7jXjv5UvNi44nhjRcSpjARRfZbDds7AOkMN9l5G9vxBZbVwrabjsFc7XZODA652g18vczGbqhR6b-ZVk2w1cA3chEDXJWJWwBGw3rxEKP6wDmRZfeDLut6wIC4j3mTeCHUv-PKK-SmkGgjntA7gG-BljSEONnGEOU7BB1rfhSDgDEqX_YTT4w3rtbn3-NAzrbIshnl_TVYqirbbWh6b3e629s7GrG3ABlJfnzUCY-KiJj0gfU4amaj07pBHDPzbW3k',
                    'e' => 'AQAB',
                    'x5t' => 'IBO6381r3QWOObmNaxF36HBgO5M',
                    'x5t#256' => 'yVWIatQnBpbU9lUGZmRlGg2bldGtJPpqQXfq3LhBq3M',
                    'x5c' => ['MIID2jCCA0MCAg39MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyODAwWhcNMTcwODIxMDUyODAwWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCwvWITOLeyTbS1'.PHP_EOL.'Q/UacqeILIK16UHLvSymIlbbiT7mpD4SMwB343xpIlXN64fC0Y1ylT6LLeX4St7A'.PHP_EOL.'cJrGIV3AMmJcsDsNzgo577LqtNvnOkLH0GojisFEKQiREX6gOgq9tWSqwaENccTE'.PHP_EOL.'sAXuV6AQ1ST+G16s00iN92hjX9V/V66snRwTsJ/p4WRpLSdAj4272hiM19qIg9zr'.PHP_EOL.'h92e2rQy7E/UShW4gpOrhg2f6fcCBm+aXIga+qxaSLchcDUvPXrpIxTd/OWQ23Qh'.PHP_EOL.'vIEzkGbPlBA8J7Nw9KCyaxbYMBFb1i0lBjwKLjmcoihiI7PVthAOu/B71D2hKcFj'.PHP_EOL.'Kpfv4D1Uam/0VumKwhwuhZVNjLq1BR1FKRJ1CioLG4wCTr0LVgtvvUyhFrS+3PdU'.PHP_EOL.'R0T5HlAQWPMyQDHgCpbOHW0wc0hbuNeO/lS82LjieGNFxKmMBFF9lsN2zsA6Qw32'.PHP_EOL.'Xkb2/EFltXCtpuOwVztdk4MDrnaDXy9zMZuqFHpv5lWTbDVwDdyEQNclYlbAEbDe'.PHP_EOL.'vEQo/rAOZFl94Mu63rAgLiPeZN4IdS/48or5KaQaCOe0DuAb4GWNIQ42cYQ5TsEH'.PHP_EOL.'Wt+FIOAMSpf9hNPjDeu1uff40DOtsiyGeX9NViqKtttaHpvd7rb2zsasbcAGUl+f'.PHP_EOL.'NQJj4qImPSB9ThqZqPTukEcM/NtbeQIDAQABMA0GCSqGSIb3DQEBBQUAA4GBAIAi'.PHP_EOL.'gU3My8kYYniDuKEXSJmbVB+K1upHxWDA8R6KMZGXfbe5BRd8s40cY6JBYL52Tgqd'.PHP_EOL.'l8z5Ek8dC4NNpfpcZc/teT1WqiO2wnpGHjgMDuDL1mxCZNL422jHpiPWkWp3AuDI'.PHP_EOL.'c7tL1QjbfAUHAQYwmHkWgPP+T2wAv0pOt36GgMCM'],
                ],
            ],
            [
                __DIR__.'/RSA/DER/8192b-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'q5lcEwG8rUflI1aL6omAaF5R1DFkCMllFaQ3HUwlwCWYNNyKxF1G2e-P3Y6SFWyp0sFfmDcvuebOY_Dw3KlC756bQUMEXH6TaubYDcXaKDyrdKgCSoufjhwHkNpRz3VxpkLADJQIHdijes2JN3daGARxSJLcjoSaZvq_LBCIHTTDGESBXJP6RtbjAjGjuLUgmcvkl029Xl8ylkrcibjTzXmOod3vioTnX5aZNT1c7evmskvixWG1NlHOhZ1HdXiPHLjKxnr4lHl9lxtTjkNSsF-Nz0bYHCpWZ9u98nkgvFAxNUmiwX5nHIqo39AK8YVuVmDGYzY-dPtD1UtCBXgj-Ryq1cPU66H7kEfvbn1kZRF0XcxqIUVDlpa_h4Aq7r8KnQ6nVF59oM8AwsrRu3llvlRzNCaRUhafZ6YUHR19j_6GpAJtOWwwV5m2qKs9EhfL9Kvz9DqWh3DBt0CuGIDS0NuBAt2_RmNQBP1u7L8pYZ_9kV-Y7YM9ocbuYpUbTy4vio33Pl2wG8iozgPIgOcbne4Vh4TGpe0hbXaL-a_84CVOWX4JyLxyBEWGB6PLfH74NyXyvh57X6Cn3P0Xr2rSgPEgEEovw5i9qDeqqzeO-GvUouhQjZgURP5USjd120IPjVoZP8RPRCAPUBQSUmi2dyHANRI3ydIhTKOEdZCVvIlVNu33wfN55kEeLCXBiDvfvozUbCGuuOkbs5Yz7vE8K9xlU_Xo2icptY_u3XMPW6YKRP6lvGtovn9295vENHOJDFCVkcJ819vHVqJnoiGAf_QX0J74NLm6fnWboH6-5BcIDl18uB3qEFAlneRflIrC2XBZju-dTuTaHy14WvVJNjTMUBgVQ4gaS1X2wmAztwv-Rk8o6k-KJuSZDWVEZyH3NaddkYSVONOMzIuuClbg4cEgLP2cxxqz8JdnyT2NNfMdGfxP4Nd_RvPtTD9kTVewlurzYVjoi8CC6VhV2Tgcp-UvT6Z0Yne-65dXi31VRqQWG8adWQ3gc9NP1oXfJqVt26ldXF9AVf7PcFcm7xzr2mwZKY-DMk1m1hBvUGeg7Iab34OABOY6J4AxXiXqKx3JV24SFydaSSevsulSrmUJU3g8TR-WwTh06Yp8DZplCU9MEvfyUSShtHED72anVRgVe8jw47k9TavJ-hPiAq0HUmmKGUeKvrqWN4bMpSiMCmHTkcqS_d4Dn4ZAI8W0DIluc9sXBaiUUSIt6t7gGNOZGUyZ-ZN4GNxVlMazB6CieGRhoPfRmXw7wq0k2R5BU1Q8PSj8jrZ88DgdfENnWCGy6Aq450OwfufGaHZDwAUD1kUoRGBkzIxvkWLVdQtmP4iZXOLSany0RtPZLGjSH-x0vQ',
                    'e' => 'AQAB',
                    'x5t' => 'YV6dSQ9sNS7rhNWcj-M4XuMmOE4',
                    'x5t#256' => 'ZNEUscWwJu03bRinDYd0BAuwiWGG3oDocehVMwX2oVo',
                    'x5c' => ['MIIF2jCCBUMCAg3+MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'OTI3MDYwNzQ5WhcNMTcwOTI2MDYwNzQ5WjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wggQiMA0GCSqGSIb3DQEBAQUAA4IEDwAwggQKAoIEAQCrmVwTAbytR+Uj'.PHP_EOL.'VovqiYBoXlHUMWQIyWUVpDcdTCXAJZg03IrEXUbZ74/djpIVbKnSwV+YNy+55s5j'.PHP_EOL.'8PDcqULvnptBQwRcfpNq5tgNxdooPKt0qAJKi5+OHAeQ2lHPdXGmQsAMlAgd2KN6'.PHP_EOL.'zYk3d1oYBHFIktyOhJpm+r8sEIgdNMMYRIFck/pG1uMCMaO4tSCZy+SXTb1eXzKW'.PHP_EOL.'StyJuNPNeY6h3e+KhOdflpk1PVzt6+ayS+LFYbU2Uc6FnUd1eI8cuMrGeviUeX2X'.PHP_EOL.'G1OOQ1KwX43PRtgcKlZn273yeSC8UDE1SaLBfmcciqjf0ArxhW5WYMZjNj50+0PV'.PHP_EOL.'S0IFeCP5HKrVw9TrofuQR+9ufWRlEXRdzGohRUOWlr+HgCruvwqdDqdUXn2gzwDC'.PHP_EOL.'ytG7eWW+VHM0JpFSFp9nphQdHX2P/oakAm05bDBXmbaoqz0SF8v0q/P0OpaHcMG3'.PHP_EOL.'QK4YgNLQ24EC3b9GY1AE/W7svylhn/2RX5jtgz2hxu5ilRtPLi+Kjfc+XbAbyKjO'.PHP_EOL.'A8iA5xud7hWHhMal7SFtdov5r/zgJU5ZfgnIvHIERYYHo8t8fvg3JfK+HntfoKfc'.PHP_EOL.'/RevatKA8SAQSi/DmL2oN6qrN474a9Si6FCNmBRE/lRKN3XbQg+NWhk/xE9EIA9Q'.PHP_EOL.'FBJSaLZ3IcA1EjfJ0iFMo4R1kJW8iVU27ffB83nmQR4sJcGIO9++jNRsIa646Ruz'.PHP_EOL.'ljPu8Twr3GVT9ejaJym1j+7dcw9bpgpE/qW8a2i+f3b3m8Q0c4kMUJWRwnzX28dW'.PHP_EOL.'omeiIYB/9BfQnvg0ubp+dZugfr7kFwgOXXy4HeoQUCWd5F+UisLZcFmO751O5Nof'.PHP_EOL.'LXha9Uk2NMxQGBVDiBpLVfbCYDO3C/5GTyjqT4om5JkNZURnIfc1p12RhJU404zM'.PHP_EOL.'i64KVuDhwSAs/ZzHGrPwl2fJPY018x0Z/E/g139G8+1MP2RNV7CW6vNhWOiLwILp'.PHP_EOL.'WFXZOByn5S9PpnRid77rl1eLfVVGpBYbxp1ZDeBz00/Whd8mpW3bqV1cX0BV/s9w'.PHP_EOL.'VybvHOvabBkpj4MyTWbWEG9QZ6Dshpvfg4AE5jongDFeJeorHclXbhIXJ1pJJ6+y'.PHP_EOL.'6VKuZQlTeDxNH5bBOHTpinwNmmUJT0wS9/JRJKG0cQPvZqdVGBV7yPDjuT1Nq8n6'.PHP_EOL.'E+ICrQdSaYoZR4q+upY3hsylKIwKYdORypL93gOfhkAjxbQMiW5z2xcFqJRRIi3q'.PHP_EOL.'3uAY05kZTJn5k3gY3FWUxrMHoKJ4ZGGg99GZfDvCrSTZHkFTVDw9KPyOtnzwOB18'.PHP_EOL.'Q2dYIbLoCrjnQ7B+58ZodkPABQPWRShEYGTMjG+RYtV1C2Y/iJlc4tJqfLRG09ks'.PHP_EOL.'aNIf7HS9AgMBAAEwDQYJKoZIhvcNAQEFBQADgYEAiXbxR0T+C6MT7Jh/SbDJ/1Gd'.PHP_EOL.'vbqskiKmmKnzOcX1x0uUHY4zHIhx3M0neYRr//XOh+FeSwM1JqAPztHy3SMRXzfP'.PHP_EOL.'xzm/nwbRwdK8C/fPy7H+uMV1mKumem8WSoOMOoxFJ+o2nJgyViwnEOu9EejlH1sc'.PHP_EOL.'uKPIoTCLUCInRRhrI84='],
                ],
            ],
            [
                __DIR__.'/RSA/DER/16k-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'tS3aeWW_wzlyXsDNFeBONFNq7W4lNWDjOUseNxx-R9AsqNJEWZFzaTtBI4Cam9Wf_2AlfP6i3RRpK76ooZObKwJmm1ReGcP7gf7JnODQv0W-m9x85a_fwHiI86Dhfy1YNh2zg1DO1kL_Q-sqKMOZ4g6uUfXGXjS5968sKCua3o-GEr-7GM6uw8zgpDmURtpupAFj3X1qCg6cjblPzMzcXdjACP4_zJpLc-sWpqY7pdLa26J5dgFGpTKWS7Xs96AlCPDz4uTRRFKDZarMFtzpjhWhNZyDGuYFFxNL4ca1tm-r4JyL-XuK9BTXC1WNXpqutzHNOj-tO9nCtRX02ZS3hmm1A9xndTZpfQ7lPuSA_kZEohkjcGyxtS-nup9khyMKGwvhg0MJS43VOuYSV6msk_z4dZ3-MCXVlJMTxLqWOSGHxHG0vDJQI5_IXCwkQLrVQIbt_X1ZylUdkmnKm4VuCBt4AHqK1F1jWpNXLYcFY-QW43c2Iln7v1uQFm_82CFHTanrNMBYNax2egYpSXpPS0naF6O1Y8bMPjPBU1jaoBAlfiSjCmHx5MOTg-PU9m1OnnR4XnOdDR0W8rUSS_iYz4Ucivou_7_XCTVlfuieAXT069ibXpGkTE58AgI6piVVYtaxyoADb3zr0a11Br0kS3gKRqxTq5GtgWTpz75VrFxXk8ATfwZF4PcOVX9fkUQStBKY9OGRryswLJbQ0lnz5ZR8QAAw1D2cAoWYoxUol5upjsrYrsGc7Ol3NhPPtoE0Vnxg49xQSZ0hhjhryId07aZ3QBr3NQ0XBoGhSNvO-M7ZyzDTNSUQusS5WyZsZputw_12E5y55_bbMPpKYbnx0aG93wmcna49jXoo5ZUMoJ_BQLaavGC0u-JTLT_subk5tKh3oVgDRevHwYeK1U4L4cdzpOzG8JVpcHyclofESk25DnHiQ92BfB1DR-6YadfQ8a4BXrIFlktDho1EmhXwWdT-dhO4ERPwf2Cx04iP3OdYCU_TNr3gVdB3fQLPkhV1K_od8zWrX7oqCGLkYCP_GUvl84dJoMequlyIO9IHtVpVHzGl-E48JoOHN00ULnoHzWzxUeGtda4twn9NQ-ptEjy9u0_8R-y2UqnC632wEmHpHzFqrOSYixp4GO_zAh-gmIhPJHuoH97vdcDRjGGFPO7kmMI0tBmxkt03ahYIqJKbPynHVLhsTuU7TVYrgTX6JkCR_IbudQTqVdz8oYO6tNqVrU89JI94_5ndJX1Wjmf1LPa8c31IQovBB0e-MlZ-rBkyTEttNuI8xC__OycsLhjAFx_bm0Qf2jfg2IJdLmDjGFHv3RlEdlRmJSyLY_lqKV4GAhjiEIEmduAKbygg2Jqhb6NKzHr1vxhRcWasnuhgTOunlGs3vezu9xz_4CvEKRMT6viU3tzqmGpT3zE7d0w9zMwn2eUlX0j7pKIiznrbkW2Dfe63f9X9bKYAsO5pcqcfAHqVaHl0iFXy5QoFwwjSuWwxKyhaY3tfY2rufLXCOzQ_G7BDoMRns8x6nCR-64Xuvp-EvBw0S790J_u9Z2W98rrW6c1cfn4cb9BRy3Rj64kWqlAUTRu6-qrX2RN5ywhoKfiJDH3m2q_MtgDlR3ke-5KuxaZwfM0hrcCppU5THbOwMe3XoDX-ZjD-3q-ikM8ueu4uTqDjtQrTQioFIxa-3prbNTsxBERQFZwlJtz2GmNHEAjgU-OwkMDObYAGc-ZAZritXe9vGtGFpdowMZ5k0FTUKSIsecPxn-nZlG-_qML8S63NXlU0RdbtYaLwQteFuXl_acAvuxOOnB3nZppJyIStP0uOPGhRowXSlThn0yFDht65TLly171JVrf4oFBDO4Q6EIJ7JMbRXCaEWJmeSNe_k71c3u4elbZ-C2i5JaO6bctZzO-xZ-CP7raQzHXMlpChYXqmpDU5bK2ySAbcDJDvg5WeRmQsqRxsFnI1EK1Jj_BKHZqOPz_q2SYyv69zPTsp5_w9z9YWCbOKP1KHyf9i9n6P5G3QkCzvlTDAbjR6nrrrnva0PZ0SjO4MzDOsIAa9S6vwRnWyE23vVI5RCv-IkLZ075LRkXKcj2EVPrDI3Mb1pUtfGu1H1M7m6V0SOTnaCwimIz3Ju2mwgnR-2lAAJKMd3vUaN1NfbEDuhZoMZfDrWzqOqA8Z2oyv6jHhby3DknbW4pn8tcaPCvScn1wotOeNFDvAfOIxikGEjUuXj-_gV2_dcFVIANseYpdhAS1tJjVX9JBwWcjbHnShy_9Y4f1zzrSwv4UbG7xDEGS1VaDUk5UwTTeeKQKzCkd6nYXxZYRMYDD_DcuGiCG9YvWm9hry1DkfdyCx5Pe8j8KMGUuEtIwLOIfmJDiCmE9fRAY85f9TAXyxkM-P1S_TcScKeSYrxLubX1DTuOGNDFKB4xNf0vi-lCFgLmx8tOr-RY7qtzrwrfcf7Kbpop-B5KpA2PhmoRTtZl4kF7RDeh-ZnUqcfyQcCIv_HewiMOmJ6iQDDjOWbsM8uEhl3ab-FzDYGAeT5bJs4EJAwEhsk28sXqnGzwJDUxw4mdLCYhiuI0ZwGOBUIzXQ-KHaH88PwYuQGLwM-s9uCKqJyO84I6XPe6bnqsL9NWmPhpvFxEZ7MvFCC2Z7nuGswQKpL92_QOS0NibZwzxBDUY7Qm3WsHxFzYm73JenQJGKzZPPLtjhIar7af2qb8KINgWGfIvrxR38osLT0Vg29M0DuMc',
                    'e' => 'AQAB',
                    'x5t' => 'XC_s0q4lqNalTFU6tNWR_Szk5dk',
                    'x5t#256' => '3nz2wIAoSbfVCmvy9k18bCPyIXacd3YfHrGq-qg3DVY',
                    'x5c' => ['MIIJ2jCCCUMCAg3/MA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIw'.PHP_EOL.'ODIyMDUyOTAyWhcNMTcwODIxMDUyOTAyWjBKMQswCQYDVQQGEwJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wgggiMA0GCSqGSIb3DQEBAQUAA4IIDwAwgggKAoIIAQC1Ldp5Zb/DOXJe'.PHP_EOL.'wM0V4E40U2rtbiU1YOM5Sx43HH5H0Cyo0kRZkXNpO0EjgJqb1Z//YCV8/qLdFGkr'.PHP_EOL.'vqihk5srAmabVF4Zw/uB/smc4NC/Rb6b3Hzlr9/AeIjzoOF/LVg2HbODUM7WQv9D'.PHP_EOL.'6yoow5niDq5R9cZeNLn3rywoK5rej4YSv7sYzq7DzOCkOZRG2m6kAWPdfWoKDpyN'.PHP_EOL.'uU/MzNxd2MAI/j/Mmktz6xampjul0trbonl2AUalMpZLtez3oCUI8PPi5NFEUoNl'.PHP_EOL.'qswW3OmOFaE1nIMa5gUXE0vhxrW2b6vgnIv5e4r0FNcLVY1emq63Mc06P6072cK1'.PHP_EOL.'FfTZlLeGabUD3Gd1Nml9DuU+5ID+RkSiGSNwbLG1L6e6n2SHIwobC+GDQwlLjdU6'.PHP_EOL.'5hJXqayT/Ph1nf4wJdWUkxPEupY5IYfEcbS8MlAjn8hcLCRAutVAhu39fVnKVR2S'.PHP_EOL.'acqbhW4IG3gAeorUXWNak1cthwVj5BbjdzYiWfu/W5AWb/zYIUdNqes0wFg1rHZ6'.PHP_EOL.'BilJek9LSdoXo7Vjxsw+M8FTWNqgECV+JKMKYfHkw5OD49T2bU6edHhec50NHRby'.PHP_EOL.'tRJL+JjPhRyK+i7/v9cJNWV+6J4BdPTr2JtekaRMTnwCAjqmJVVi1rHKgANvfOvR'.PHP_EOL.'rXUGvSRLeApGrFOrka2BZOnPvlWsXFeTwBN/BkXg9w5Vf1+RRBK0Epj04ZGvKzAs'.PHP_EOL.'ltDSWfPllHxAADDUPZwChZijFSiXm6mOytiuwZzs6Xc2E8+2gTRWfGDj3FBJnSGG'.PHP_EOL.'OGvIh3TtpndAGvc1DRcGgaFI2874ztnLMNM1JRC6xLlbJmxmm63D/XYTnLnn9tsw'.PHP_EOL.'+kphufHRob3fCZydrj2NeijllQygn8FAtpq8YLS74lMtP+y5uTm0qHehWANF68fB'.PHP_EOL.'h4rVTgvhx3Ok7MbwlWlwfJyWh8RKTbkOceJD3YF8HUNH7php19DxrgFesgWWS0OG'.PHP_EOL.'jUSaFfBZ1P52E7gRE/B/YLHTiI/c51gJT9M2veBV0Hd9As+SFXUr+h3zNatfuioI'.PHP_EOL.'YuRgI/8ZS+Xzh0mgx6q6XIg70ge1WlUfMaX4Tjwmg4c3TRQuegfNbPFR4a11ri3C'.PHP_EOL.'f01D6m0SPL27T/xH7LZSqcLrfbASYekfMWqs5JiLGngY7/MCH6CYiE8ke6gf3u91'.PHP_EOL.'wNGMYYU87uSYwjS0GbGS3TdqFgiokps/KcdUuGxO5TtNViuBNfomQJH8hu51BOpV'.PHP_EOL.'3Pyhg7q02pWtTz0kj3j/md0lfVaOZ/Us9rxzfUhCi8EHR74yVn6sGTJMS2024jzE'.PHP_EOL.'L/87JywuGMAXH9ubRB/aN+DYgl0uYOMYUe/dGUR2VGYlLItj+WopXgYCGOIQgSZ2'.PHP_EOL.'4ApvKCDYmqFvo0rMevW/GFFxZqye6GBM66eUaze97O73HP/gK8QpExPq+JTe3OqY'.PHP_EOL.'alPfMTt3TD3MzCfZ5SVfSPukoiLOetuRbYN97rd/1f1spgCw7mlypx8AepVoeXSI'.PHP_EOL.'VfLlCgXDCNK5bDErKFpje19jau58tcI7ND8bsEOgxGezzHqcJH7rhe6+n4S8HDRL'.PHP_EOL.'v3Qn+71nZb3yutbpzVx+fhxv0FHLdGPriRaqUBRNG7r6qtfZE3nLCGgp+IkMfeba'.PHP_EOL.'r8y2AOVHeR77kq7FpnB8zSGtwKmlTlMds7Ax7degNf5mMP7er6KQzy567i5OoOO1'.PHP_EOL.'CtNCKgUjFr7emts1OzEERFAVnCUm3PYaY0cQCOBT47CQwM5tgAZz5kBmuK1d728a'.PHP_EOL.'0YWl2jAxnmTQVNQpIix5w/Gf6dmUb7+owvxLrc1eVTRF1u1hovBC14W5eX9pwC+7'.PHP_EOL.'E46cHedmmknIhK0/S448aFGjBdKVOGfTIUOG3rlMuXLXvUlWt/igUEM7hDoQgnsk'.PHP_EOL.'xtFcJoRYmZ5I17+TvVze7h6Vtn4LaLklo7pty1nM77Fn4I/utpDMdcyWkKFheqak'.PHP_EOL.'NTlsrbJIBtwMkO+DlZ5GZCypHGwWcjUQrUmP8Eodmo4/P+rZJjK/r3M9Oynn/D3P'.PHP_EOL.'1hYJs4o/UofJ/2L2fo/kbdCQLO+VMMBuNHqeuuue9rQ9nRKM7gzMM6wgBr1Lq/BG'.PHP_EOL.'dbITbe9UjlEK/4iQtnTvktGRcpyPYRU+sMjcxvWlS18a7UfUzubpXRI5OdoLCKYj'.PHP_EOL.'Pcm7abCCdH7aUAAkox3e9Ro3U19sQO6Fmgxl8OtbOo6oDxnajK/qMeFvLcOSdtbi'.PHP_EOL.'mfy1xo8K9JyfXCi0540UO8B84jGKQYSNS5eP7+BXb91wVUgA2x5il2EBLW0mNVf0'.PHP_EOL.'kHBZyNsedKHL/1jh/XPOtLC/hRsbvEMQZLVVoNSTlTBNN54pArMKR3qdhfFlhExg'.PHP_EOL.'MP8Ny4aIIb1i9ab2GvLUOR93ILHk97yPwowZS4S0jAs4h+YkOIKYT19EBjzl/1MB'.PHP_EOL.'fLGQz4/VL9NxJwp5JivEu5tfUNO44Y0MUoHjE1/S+L6UIWAubHy06v5Fjuq3OvCt'.PHP_EOL.'9x/spumin4HkqkDY+GahFO1mXiQXtEN6H5mdSpx/JBwIi/8d7CIw6YnqJAMOM5Zu'.PHP_EOL.'wzy4SGXdpv4XMNgYB5PlsmzgQkDASGyTbyxeqcbPAkNTHDiZ0sJiGK4jRnAY4FQj'.PHP_EOL.'NdD4odofzw/Bi5AYvAz6z24IqonI7zgjpc97pueqwv01aY+Gm8XERnsy8UILZnue'.PHP_EOL.'4azBAqkv3b9A5LQ2JtnDPEENRjtCbdawfEXNibvcl6dAkYrNk88u2OEhqvtp/apv'.PHP_EOL.'wog2BYZ8i+vFHfyiwtPRWDb0zQO4xwIDAQABMA0GCSqGSIb3DQEBBQUAA4GBADg3'.PHP_EOL.'1Ah8wT/xjIhtUTAIcFGtgN2321aV8pIz8VSJu3CrbJJD09Ek6WUQgTbEq0pxwhQo'.PHP_EOL.'ubkr2+CJ2Gw/FTd0WFet7T57aFg7qh5xraEhH21icHmNBUG7ETUXNEf8TjbhREVY'.PHP_EOL.'gF6l8RI6rrGv0zm5awmcj+4+2OXQ+OM88dV7chMn'],
                ],
            ],
            [
                __DIR__.'/RSA/DER/32k-rsa-example-cert.der',
                [
                    'kty' => 'RSA',
                    'n' => 'qzPFsFIf3cSes25DloV3y3d8gKMcZVE_EQ_6e_MZnyqDbuOEP39yQs3aunzbZRoO8Xw8lLoJNduiKKsco7odI753kBvz1eLyke-sWBVZttbnYyz9AE3ZXfAb9rHW2AxgIqHNsQOsLJS_douGZwxawNdE90WM4QG80bDpkxxHfObtmZIbZoOFSeokDHA5jokQGzJ65t6ARtQOIht84pIlAr8RO0vCUiJ0R4TdAffbdIukMcVfSoZBlZJ_q-yBtPoqB1Nmr1x1FqCtR81NrEtdp7CUHy4yLIskMzHTwJL24dx8zPS9RBIAuR6HO6soQwQgKY5NYmyaZGuWDrzw0Lor9_jjcx3x7NlXEUffGyUdT_bZ6owsgd-SpvnbqXPXIf-u5JH7afSUuajytHnGVilQOpEg06B0F-AumUEx8vdLPczCx0CED11mhRhT1eRQPJlzxgqA22SN1Yz0P55R8QbfFYcflpEtZbHmdvwMSipEoEUyI8aA9z268oNVnnAGhG3cOqk8-4HOvtqZ9LIc8jUcQLtWX-PJav9EePnWuV6pFwzvKcwl09m08xIfIh9DvFVJz3Fks-X6c1tVo2Valftlj8fnlzu9WgownkwhM4KN2UpcHcff4G-v9zckhcpROSzZ1ax5mPOUMF6B2OVawMhf3li9A9JEpBDxVu2-gZU6NbhvfH1f4PdNPUnlasPylHn4qz4S6_V1fuxho-2O_V72w3V5FDBi-m2D9vDVQvJtuoiJxUEyOWaxsenuzoFlq3jNHwm0SiabwVjaMyre4qktmHopLuLX2ixME3rbTtaXLAaly-t2X6oS4nFyhwP9f_WbJb4Yh_RDxksPj1hR_4nH43NTYjZBlLDM0YRb4xRzFmATQOUhPou6LSUbl8Tl2z7WYFzlcKgHwkWRaTGUV8Sz_h-_IfgZDvCtyyLhzvWOmfJBhsV1nTbDrr8DivZGH5huBNH88v_gbCVw36aAjH8BnmcHQ0ImUUwXoiB1iWSWB3x1xdYnAyQf5RV2PK86wVc4EBRxW6MeJHWZr-kFgHtcwk2ys8MewL8xlKs1S64APAWtD-WsLGEnUVMfM5EuWjoS9kB4BI4DC6M0uyDjaCuFu80wMmWfx9C3-Y2x7l5Lw0G4gRcUk-F3ONtKfsxMqAmV6aUVkXmdkX5LLa105CpIVqflM40CPl5mlVGEFlTf9u0zclyQ0_-qWt78ZzkpolPj9XKHikdYA_DKbvtfgtgNC07GIwBctoQsOrKGOxigeWzrAwfS9S5Wt7hvcs2R0Y04rXoeSTPbHWLumsJYLxC2HPtam3IxQJzCljIOFB5Sqi9WLO5l_yjmUGS2Fzy5DkuyFuC3o79rB-Vu0zpHQ5sHdbyYkfvi3QZx4jLuj2ki-3_1Qj7RfVdd1yWeudnFUy5QGfWh3-VoaK9UIZ1EeX62owXTGNOJovn9yMdwbXmy75qrkPXadFQG3lnuqq_Ucd8ZAYJvwfQb6uhTSv1kSFCpxyyaSBYjLU44QDF6FRh_QHLMBM2DVasOT0hsF2UWsIXUneoJHk_qVZSRmj5EDaIrWAUEZfL_geiwcW3_L3Y9iaHMkB93fHNsVEpLmTO-vLHZHYN0c-kKNVBw_40xGZ5ZgPJlT4JZVvBKuB2ka2OsSLcRXZvzZZZTnrRHb_9dngGkFpI0gc6gFu2d1mPIIFp6JS7AJ4_sYKE4yxuGG7IsA4ErnNBEK9Sr1XSu0_KfcIv63dm_AybDg1vmqMLCl5EiP9OIFsWdIM42970PH9h8Ri7KUn0D53RSRVkV38NW312A2JYCHfEfbIxyibEIrsusib98x6Bedh-3BpsWyih2XlDT6AFwJdD0cc_Uf56Vqv9waUtsSx-1xBwliZ35MKq-IfV6hcLnFgLhxsqakV8aFLAEzI8Ulned6zjRAC28aaDOZcFdKEMD0wHPUW8-9UTQxAgug8otEITWSkKubyXbdofpVa9Xwjq1-jLb4eylqey0RokKrHO6B7F3KtUF8Zsm0mGEg7nvUhjEBFL3AqkLke5Nb_78uqb3tzZF3iO6ghENar9s1DUIYqNkbMSeh7smgER_PBUB0MGMqRnx8qcr5t5yBEurZ7qq7-LYoJOoc6UwaPrQN_AFRou4ugiRrxIrvOwrDPr4y2zoi9XKnBBuYMnt2AkGVCNIA0WOKgmex4x_2Nri2JlRieAPwNPfW5PLkyPVRfw0dNzhg7csMl1Wctdw1JpHJhgMswuhYhRWGyzYWE4ZU8lvQWqA42MOKfUixAV4LmEzGz2eRQSPGWjLC85-mcxf_vssmD-mbuJAjzlLDzzwllrTDCQrt18DftpAAHhD5hG2HmQH9RDzcS3sniIx4p2zyqBHVQsWM74BlQjbODjgHRHerTgxYaNmh4KRA38lmb9omrUhI2Q0Lj5CF2of_Apd7fo8u6LpBpdEtirkn_7-9vPPiGerClV6lSjoNi_I_hHCneAq-3KZq7hM5XliJPvUrws_m0X5n6_fazdk-gOohEuF0Aq_1I5633sS-DGrFyan2K7oeoBGQN994-kweTR0lLko14nC5wnvizbsv7sDUNJTjM7LMYIrhKEILTjjGQ6WuCkYhQuM4RAnx74jFIchW8pS1tEnUcIOyBWgFB9M2zdbNmJg7vH43mmX408jMYVKs9CQz2Y7Vu33S0dSp9sWxM1KUREFVy1xTbVgKNxLxOzXiLOjm_b4EifAHZh_KTf0POm5RESU-TSrO29y5puTHL-PLuOE30jrxXaKhW5UzmQLUMhBGI7geYP6fE6QxyUi0gD_tLdMmzxTlZiOXkE6HnBQ-3Ar54uA-RFUhnzU-XT3wm--eINsvqyrHCyLQlmM71aBXnMlH5g0NJjdm42XSecTopWfFCfcNe1-ufpUuMGGg0C3LxVN5fkTmB2_6gai0AHh4dNhefGkKCZ5OcSNtA_UUI1nKr_wgPTI4X1catN9RE9mMYhOt-I5gOVRCihxDcUcBl2apUaFK-jHPs5rABqhykbi_dOS-zy42I86Vcu4B-_0GNlRIPRLZWFIhNRy_kfCOq4kb4SK9DjTvHsaq6YWMoL9Jk3JiqvH4yrMZ6T-XEFdJ8DGSc41lo1YJwhFUu0eGbGFKxyUBrHv1l9ByPrqWaiepnBBsda4y8G3SoiCfndwkbvLeE5ykYgurPpkYX_bau2PqsoAkiJ_GmbitKpXD71C5PmzvzLvpxkgC6hQq-v4L4WLelADvBpeikX9k23qhR5H3mkzNeMZgHyoFisy161cDgOlcg64g6C2UzJKlb5C1tOlQwM3fdm7cjBJXOjuxgi8Ewx6ov90eeaqIEfFvnUu1_IC_tFve9P_Us21Ak53vwStlHueYHtedJsHg84C5Ppt_z1LFR3Hh8m1pOnlb3kJw5eGpvsXweZrIIN0cvwz-NZ_orIxjPxLf23wy-y-lhObK17BfX1g-p759XtRSaG4Rj_QedauXHAA-SKgvwAOY3kBuWo9Oxx73JbC1kov55TkecHj2lXO_o49O5LCOa_h0nHIVb3JIGWot11sF_6zwNzFM2WtHFNu7Iu9hllumC8rvz3HEbylvSPQYzBQKy8NSyC6T9wbH6cAYY-vl59q1J4DwBH3DHKoMAec8InlnBO_ekJa8SMdQMZxov0BaxJc0W__29w2Sza0cBsMslfpRIWRWMb4jNpyvCyEVxrGf7AakOl0_9P3JCQ2o8cuf-BGg_z_iQ3aTMYVWi_pWuxnhh5NchjQU8C3dxvnEd0Te9mmDlvZh-N9GULo0tlzHz3WZniUp7mxVQ3nkeS31M0LIIF3SetSMjXrGJ_4bzAnb3EjH44eFuvgOiJ8ChXLCmHLtIpFa0WSC6YVpBxqfPrxke-DyB2Lvz_46MSQ4iKvCFhdYWxBtwXCZDN5Dt4XFpMknL_VnuVU8a5_rRqpEebv_VF1pBZsvfTK6UXFWAApFvL4ebApuLsFInG3uk89N2SbenTTiBGWZWZjsEFsvf3iSFZdQ2bgKSLmJIsuXV1mUPkzGEr8SsPLDKhGNZBevtka-CfnukEPn7a3K_O5sYcccEtYwx0VNiC6dWu7B_-pflffa1m4pbhdg6KfykDO9_jU_LE692dhWUzbv977zGUlOnmsEMeqmSTo9V5Hv0UsEDGEjoe9piKidoZ8JdAq1WIpSBfW9M2wtkZHbi2nlaBnKJuTaaNs_nWjbG4y73hEqEqRlQMKrLsJU7rsmy3h6x6-J_tXfkKpWu_Z_PhR-ca2RV4ldwUNejBhBomg-6bcSq1lHXGTpwc0wSDmIUfE2W6ZZysaFpmGpTDFjTDqfeeAwwbzShK7Uc-OnJVNiQ5w1KALJNjXURSfI61vyWRBMtFHaC7t6ixwDfv6pqEa0xeDe4xf4Z1qdX1Zfs4xpdAyzZWmslUsXIYDtiTXq6NYGjnCEPYqneVGOWhP6re0UfzeqqB6p6_L42UoqFrrjU7jnEWRlz6gxdU9qOJgLX3u6CIYtN6b44tpsqA23fNBiuf4SqoYimbd2YVjXFRFFNZ2XqJ-wBqYcD5xIfudMN6W5cAD4p5cTQ11_-EqIp8rDxiWOs-PN8SQTIE7ZYQ6na-lSITpchNybreE9SqhzluoY71DN8oQuUJHonrAW5Hh_VroGBxpbO9XdNhw0XrC-S9iH9DDEUedanM2DznPUZsHHutG8H0K9AEyWRS01sAwrF73ZG57qy5IciYMHZuFbkY0lzwbF-vd15jgNfP4JTmZD2sVWwVgI7Qp9T2hd0uuZL_huHl2baRCyC_DSI9c6p3q9Ud_tBN_yCcNcUVx0rS6EGfzM8VYOGwyiBVBAgVDjBXiKBsUVWA3ljfOtYhLKBDHkqhvoQaczSI2fKX7L7cwgXeBdckoaNhno6mCpZBamuyBZ1Iy6TnguQi59MCCKdiczIpfeumbSDEovy2IbQmPqld_JI6WOufgldiITu3hXR5KNazan2mc3NrKu1SEXZpdzb4wJZZ26U_1xE2GLMJru05yZoVNEkN72DhagM1R5oqHwPzRcn3ahdYvUzDoP6UHEpa76A23lqafY7F98l66hmAnXXlEKzEVwthYoxWANYtVsxs9NktNJdNMB3OCMnCo9BWkefmjlrzMJSkBP_1mfxN2o3W1tMNXpk5OQPO20_eWPF3iYhobSo8fcxzXtw9bg1BXr0TADj0hl_z4jw93wVGGLlsA3qYstay0I9yJgHBZmhxc7V1JzNWdwxIDmRgA5eCm1ELVBxpIup9WGZlUs1rzwqXzI-37i7l3dwFfCf_i2g8m-gNQjuM6YqkSz-XKcn-sJEg1XSMhoB15sgYE9U-2Oe-_EGLK0dOU2zyHO40F8ghvhKWpuAcITX_QnEMremwsiCl0PEnGZ98BXzlRvd1MFNc0ZUwzN-wTVxs4jNkteNbp0MjIKA5Y6FiCEX6koNWY9cLXSNg4XG4IsWRQrfIn2WWFz_nhzlaZNm_NUM1kmKRREPmsvQ',
                    'e' => 'AQAB',
                    'x5t' => 'KGApLybHWJmBwZGgBk07AlRD9nU',
                    'x5t#256' => 'YD12k6kc4xuh_5vEHMyyOFpGs6VqTyaKMlxg0Nt2crA',
                    'x5c' => ['MIIR2jCCEUMCAg4EMA0GCSqGSIb3DQEBBQUAMIGbMQswCQYDVQQGEwJKUDEOMAwG'.PHP_EOL.'A1UECBMFVG9reW8xEDAOBgNVBAcTB0NodW8ta3UxETAPBgNVBAoTCEZyYW5rNERE'.PHP_EOL.'MRgwFgYDVQQLEw9XZWJDZXJ0IFN1cHBvcnQxGDAWBgNVBAMTD0ZyYW5rNEREIFdl'.PHP_EOL.'YiBDQTEjMCEGCSqGSIb3DQEJARYUc3VwcG9ydEBmcmFuazRkZC5jb20wHhcNMTIx'.PHP_EOL.'MDEwMTIxNzQ5WhcNMTcxMDA5MTIxNzQ5WjBKMQswCQYDVQQGDAJKUDEOMAwGA1UE'.PHP_EOL.'CAwFVG9reW8xETAPBgNVBAoMCEZyYW5rNEREMRgwFgYDVQQDDA93d3cuZXhhbXBs'.PHP_EOL.'ZS5jb20wghAiMA0GCSqGSIb3DQEBAQUAA4IQDwAwghAKAoIQAQCrM8WwUh/dxJ6z'.PHP_EOL.'bkOWhXfLd3yAoxxlUT8RD/p78xmfKoNu44Q/f3JCzdq6fNtlGg7xfDyUugk126Io'.PHP_EOL.'qxyjuh0jvneQG/PV4vKR76xYFVm21udjLP0ATdld8Bv2sdbYDGAioc2xA6wslL92'.PHP_EOL.'i4ZnDFrA10T3RYzhAbzRsOmTHEd85u2Zkhtmg4VJ6iQMcDmOiRAbMnrm3oBG1A4i'.PHP_EOL.'G3zikiUCvxE7S8JSInRHhN0B99t0i6QxxV9KhkGVkn+r7IG0+ioHU2avXHUWoK1H'.PHP_EOL.'zU2sS12nsJQfLjIsiyQzMdPAkvbh3HzM9L1EEgC5Hoc7qyhDBCApjk1ibJpka5YO'.PHP_EOL.'vPDQuiv3+ONzHfHs2VcRR98bJR1P9tnqjCyB35Km+dupc9ch/67kkftp9JS5qPK0'.PHP_EOL.'ecZWKVA6kSDToHQX4C6ZQTHy90s9zMLHQIQPXWaFGFPV5FA8mXPGCoDbZI3VjPQ/'.PHP_EOL.'nlHxBt8Vhx+WkS1lseZ2/AxKKkSgRTIjxoD3Pbryg1WecAaEbdw6qTz7gc6+2pn0'.PHP_EOL.'shzyNRxAu1Zf48lq/0R4+da5XqkXDO8pzCXT2bTzEh8iH0O8VUnPcWSz5fpzW1Wj'.PHP_EOL.'ZVqV+2WPx+eXO71aCjCeTCEzgo3ZSlwdx9/gb6/3NySFylE5LNnVrHmY85QwXoHY'.PHP_EOL.'5VrAyF/eWL0D0kSkEPFW7b6BlTo1uG98fV/g9009SeVqw/KUefirPhLr9XV+7GGj'.PHP_EOL.'7Y79XvbDdXkUMGL6bYP28NVC8m26iInFQTI5ZrGx6e7OgWWreM0fCbRKJpvBWNoz'.PHP_EOL.'Kt7iqS2Yeiku4tfaLEwTettO1pcsBqXL63ZfqhLicXKHA/1/9ZslvhiH9EPGSw+P'.PHP_EOL.'WFH/icfjc1NiNkGUsMzRhFvjFHMWYBNA5SE+i7otJRuXxOXbPtZgXOVwqAfCRZFp'.PHP_EOL.'MZRXxLP+H78h+BkO8K3LIuHO9Y6Z8kGGxXWdNsOuvwOK9kYfmG4E0fzy/+BsJXDf'.PHP_EOL.'poCMfwGeZwdDQiZRTBeiIHWJZJYHfHXF1icDJB/lFXY8rzrBVzgQFHFbox4kdZmv'.PHP_EOL.'6QWAe1zCTbKzwx7AvzGUqzVLrgA8Ba0P5awsYSdRUx8zkS5aOhL2QHgEjgMLozS7'.PHP_EOL.'IONoK4W7zTAyZZ/H0Lf5jbHuXkvDQbiBFxST4Xc420p+zEyoCZXppRWReZ2Rfkst'.PHP_EOL.'rXTkKkhWp+UzjQI+XmaVUYQWVN/27TNyXJDT/6pa3vxnOSmiU+P1coeKR1gD8Mpu'.PHP_EOL.'+1+C2A0LTsYjAFy2hCw6soY7GKB5bOsDB9L1Lla3uG9yzZHRjTiteh5JM9sdYu6a'.PHP_EOL.'wlgvELYc+1qbcjFAnMKWMg4UHlKqL1Ys7mX/KOZQZLYXPLkOS7IW4Lejv2sH5W7T'.PHP_EOL.'OkdDmwd1vJiR++LdBnHiMu6PaSL7f/VCPtF9V13XJZ652cVTLlAZ9aHf5Whor1Qh'.PHP_EOL.'nUR5frajBdMY04mi+f3Ix3BtebLvmquQ9dp0VAbeWe6qr9Rx3xkBgm/B9Bvq6FNK'.PHP_EOL.'/WRIUKnHLJpIFiMtTjhAMXoVGH9AcswEzYNVqw5PSGwXZRawhdSd6gkeT+pVlJGa'.PHP_EOL.'PkQNoitYBQRl8v+B6LBxbf8vdj2JocyQH3d8c2xUSkuZM768sdkdg3Rz6Qo1UHD/'.PHP_EOL.'jTEZnlmA8mVPgllW8Eq4HaRrY6xItxFdm/NlllOetEdv/12eAaQWkjSBzqAW7Z3W'.PHP_EOL.'Y8ggWnolLsAnj+xgoTjLG4YbsiwDgSuc0EQr1KvVdK7T8p9wi/rd2b8DJsODW+ao'.PHP_EOL.'wsKXkSI/04gWxZ0gzjb3vQ8f2HxGLspSfQPndFJFWRXfw1bfXYDYlgId8R9sjHKJ'.PHP_EOL.'sQiuy6yJv3zHoF52H7cGmxbKKHZeUNPoAXAl0PRxz9R/npWq/3BpS2xLH7XEHCWJ'.PHP_EOL.'nfkwqr4h9XqFwucWAuHGypqRXxoUsATMjxSWd53rONEALbxpoM5lwV0oQwPTAc9R'.PHP_EOL.'bz71RNDECC6Dyi0QhNZKQq5vJdt2h+lVr1fCOrX6Mtvh7KWp7LRGiQqsc7oHsXcq'.PHP_EOL.'1QXxmybSYYSDue9SGMQEUvcCqQuR7k1v/vy6pve3NkXeI7qCEQ1qv2zUNQhio2Rs'.PHP_EOL.'xJ6HuyaARH88FQHQwYypGfHypyvm3nIES6tnuqrv4tigk6hzpTBo+tA38AVGi7i6'.PHP_EOL.'CJGvEiu87CsM+vjLbOiL1cqcEG5gye3YCQZUI0gDRY4qCZ7HjH/Y2uLYmVGJ4A/A'.PHP_EOL.'099bk8uTI9VF/DR03OGDtywyXVZy13DUmkcmGAyzC6FiFFYbLNhYThlTyW9BaoDj'.PHP_EOL.'Yw4p9SLEBXguYTMbPZ5FBI8ZaMsLzn6ZzF/++yyYP6Zu4kCPOUsPPPCWWtMMJCu3'.PHP_EOL.'XwN+2kAAeEPmEbYeZAf1EPNxLeyeIjHinbPKoEdVCxYzvgGVCNs4OOAdEd6tODFh'.PHP_EOL.'o2aHgpEDfyWZv2iatSEjZDQuPkIXah/8Cl3t+jy7oukGl0S2KuSf/v7288+IZ6sK'.PHP_EOL.'VXqVKOg2L8j+EcKd4Cr7cpmruEzleWIk+9SvCz+bRfmfr99rN2T6A6iES4XQCr/U'.PHP_EOL.'jnrfexL4MasXJqfYruh6gEZA333j6TB5NHSUuSjXicLnCe+LNuy/uwNQ0lOMzssx'.PHP_EOL.'giuEoQgtOOMZDpa4KRiFC4zhECfHviMUhyFbylLW0SdRwg7IFaAUH0zbN1s2YmDu'.PHP_EOL.'8fjeaZfjTyMxhUqz0JDPZjtW7fdLR1Kn2xbEzUpREQVXLXFNtWAo3EvE7NeIs6Ob'.PHP_EOL.'9vgSJ8AdmH8pN/Q86blERJT5NKs7b3Lmm5Mcv48u44TfSOvFdoqFblTOZAtQyEEY'.PHP_EOL.'juB5g/p8TpDHJSLSAP+0t0ybPFOVmI5eQToecFD7cCvni4D5EVSGfNT5dPfCb754'.PHP_EOL.'g2y+rKscLItCWYzvVoFecyUfmDQ0mN2bjZdJ5xOilZ8UJ9w17X65+lS4wYaDQLcv'.PHP_EOL.'FU3l+ROYHb/qBqLQAeHh02F58aQoJnk5xI20D9RQjWcqv/CA9MjhfVxq031ET2Yx'.PHP_EOL.'iE634jmA5VEKKHENxRwGXZqlRoUr6Mc+zmsAGqHKRuL905L7PLjYjzpVy7gH7/QY'.PHP_EOL.'2VEg9EtlYUiE1HL+R8I6riRvhIr0ONO8exqrphYygv0mTcmKq8fjKsxnpP5cQV0n'.PHP_EOL.'wMZJzjWWjVgnCEVS7R4ZsYUrHJQGse/WX0HI+upZqJ6mcEGx1rjLwbdKiIJ+d3CR'.PHP_EOL.'u8t4TnKRiC6s+mRhf9tq7Y+qygCSIn8aZuK0qlcPvULk+bO/Mu+nGSALqFCr6/gv'.PHP_EOL.'hYt6UAO8Gl6KRf2TbeqFHkfeaTM14xmAfKgWKzLXrVwOA6VyDriDoLZTMkqVvkLW'.PHP_EOL.'06VDAzd92btyMElc6O7GCLwTDHqi/3R55qogR8W+dS7X8gL+0W970/9SzbUCTne/'.PHP_EOL.'BK2Ue55ge150mweDzgLk+m3/PUsVHceHybWk6eVveQnDl4am+xfB5msgg3Ry/DP4'.PHP_EOL.'1n+isjGM/Et/bfDL7L6WE5srXsF9fWD6nvn1e1FJobhGP9B51q5ccAD5IqC/AA5j'.PHP_EOL.'eQG5aj07HHvclsLWSi/nlOR5wePaVc7+jj07ksI5r+HScchVvckgZai3XWwX/rPA'.PHP_EOL.'3MUzZa0cU27si72GWW6YLyu/PccRvKW9I9BjMFArLw1LILpP3BsfpwBhj6+Xn2rU'.PHP_EOL.'ngPAEfcMcqgwB5zwieWcE796QlrxIx1AxnGi/QFrElzRb//b3DZLNrRwGwyyV+lE'.PHP_EOL.'hZFYxviM2nK8LIRXGsZ/sBqQ6XT/0/ckJDajxy5/4EaD/P+JDdpMxhVaL+la7GeG'.PHP_EOL.'Hk1yGNBTwLd3G+cR3RN72aYOW9mH430ZQujS2XMfPdZmeJSnubFVDeeR5LfUzQsg'.PHP_EOL.'gXdJ61IyNesYn/hvMCdvcSMfjh4W6+A6InwKFcsKYcu0ikVrRZILphWkHGp8+vGR'.PHP_EOL.'74PIHYu/P/joxJDiIq8IWF1hbEG3BcJkM3kO3hcWkyScv9We5VTxrn+tGqkR5u/9'.PHP_EOL.'UXWkFmy99MrpRcVYACkW8vh5sCm4uwUicbe6Tz03ZJt6dNOIEZZlZmOwQWy9/eJI'.PHP_EOL.'Vl1DZuApIuYkiy5dXWZQ+TMYSvxKw8sMqEY1kF6+2Rr4J+e6QQ+ftrcr87mxhxxw'.PHP_EOL.'S1jDHRU2ILp1a7sH/6l+V99rWbiluF2Dop/KQM73+NT8sTr3Z2FZTNu/3vvMZSU6'.PHP_EOL.'eawQx6qZJOj1Xke/RSwQMYSOh72mIqJ2hnwl0CrVYilIF9b0zbC2RkduLaeVoGco'.PHP_EOL.'m5Npo2z+daNsbjLveESoSpGVAwqsuwlTuuybLeHrHr4n+1d+Qqla79n8+FH5xrZF'.PHP_EOL.'XiV3BQ16MGEGiaD7ptxKrWUdcZOnBzTBIOYhR8TZbplnKxoWmYalMMWNMOp954DD'.PHP_EOL.'BvNKErtRz46clU2JDnDUoAsk2NdRFJ8jrW/JZEEy0UdoLu3qLHAN+/qmoRrTF4N7'.PHP_EOL.'jF/hnWp1fVl+zjGl0DLNlaayVSxchgO2JNero1gaOcIQ9iqd5UY5aE/qt7RR/N6q'.PHP_EOL.'oHqnr8vjZSioWuuNTuOcRZGXPqDF1T2o4mAtfe7oIhi03pvji2myoDbd80GK5/hK'.PHP_EOL.'qhiKZt3ZhWNcVEUU1nZeon7AGphwPnEh+50w3pblwAPinlxNDXX/4SoinysPGJY6'.PHP_EOL.'z483xJBMgTtlhDqdr6VIhOlyE3Jut4T1KqHOW6hjvUM3yhC5QkeiesBbkeH9WugY'.PHP_EOL.'HGls71d02HDResL5L2If0MMRR51qczYPOc9Rmwce60bwfQr0ATJZFLTWwDCsXvdk'.PHP_EOL.'bnurLkhyJgwdm4VuRjSXPBsX693XmOA18/glOZkPaxVbBWAjtCn1PaF3S65kv+G4'.PHP_EOL.'eXZtpELIL8NIj1zqner1R3+0E3/IJw1xRXHStLoQZ/MzxVg4bDKIFUECBUOMFeIo'.PHP_EOL.'GxRVYDeWN861iEsoEMeSqG+hBpzNIjZ8pfsvtzCBd4F1ySho2GejqYKlkFqa7IFn'.PHP_EOL.'UjLpOeC5CLn0wIIp2JzMil966ZtIMSi/LYhtCY+qV38kjpY65+CV2IhO7eFdHko1'.PHP_EOL.'rNqfaZzc2sq7VIRdml3NvjAllnbpT/XETYYswmu7TnJmhU0SQ3vYOFqAzVHmiofA'.PHP_EOL.'/NFyfdqF1i9TMOg/pQcSlrvoDbeWpp9jsX3yXrqGYCddeUQrMRXC2FijFYA1i1Wz'.PHP_EOL.'Gz02S00l00wHc4IycKj0FaR5+aOWvMwlKQE//WZ/E3ajdbW0w1emTk5A87bT95Y8'.PHP_EOL.'XeJiGhtKjx9zHNe3D1uDUFevRMAOPSGX/PiPD3fBUYYuWwDepiy1rLQj3ImAcFma'.PHP_EOL.'HFztXUnM1Z3DEgOZGADl4KbUQtUHGki6n1YZmVSzWvPCpfMj7fuLuXd3AV8J/+La'.PHP_EOL.'Dyb6A1CO4zpiqRLP5cpyf6wkSDVdIyGgHXmyBgT1T7Y5778QYsrR05TbPIc7jQXy'.PHP_EOL.'CG+Epam4BwhNf9CcQyt6bCyIKXQ8ScZn3wFfOVG93UwU1zRlTDM37BNXGziM2S14'.PHP_EOL.'1unQyMgoDljoWIIRfqSg1Zj1wtdI2DhcbgixZFCt8ifZZYXP+eHOVpk2b81QzWSY'.PHP_EOL.'pFEQ+ay9AgMBAAEwDQYJKoZIhvcNAQEFBQADgYEATPINk/17H+RLz459iCVQAGes'.PHP_EOL.'8kc5sxYj3CkMlWrGMiCxvsgu2kak6dCa0f3DfiVt54Fry7s0OklHiZmipoiF4RCt'.PHP_EOL.'yJwUSAzRrZFAbkpDg8oIu4Ui/Bt13kY7xON+u4m0IgkLZSE+8BSjMrfjVvVxe+qH'.PHP_EOL.'5i7X/ibUTDjgyfdA8XI='],
                ],
            ],
            [
                __DIR__.'/EC/PEM/prime256v1-cert.pem',
                [
                    'kty' => 'EC',
                    'crv' => 'P-256',
                    'x' => 'xEsr_55aqgFXdrbRNz1_WSNI8UaSUxCka2kGEN1bXsI',
                    'y' => 'SM45Hsr9dnUR6Ox-TpmNv2fbDX4CoVo-3patMUpXANA',
                    'x5t' => 'ZnnaQDssCKJQZLp6zyHssIZOa7o',
                    'x5t#256' => 'v7VlokKTGL3anRk8Nl0VcqVC9u5j2Fb5tdlQntUgDT4',
                    'x5c' => ['MIIB0jCCAXegAwIBAgIJAK2o1kQ5JwpUMAoGCCqGSM49BAMCMEUxCzAJBgNVBAYT'.PHP_EOL.'AkFVMRMwEQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRn'.PHP_EOL.'aXRzIFB0eSBMdGQwHhcNMTUxMTA4MTUxMTU2WhcNMTYxMTA3MTUxMTU2WjBFMQsw'.PHP_EOL.'CQYDVQQGEwJBVTETMBEGA1UECAwKU29tZS1TdGF0ZTEhMB8GA1UECgwYSW50ZXJu'.PHP_EOL.'ZXQgV2lkZ2l0cyBQdHkgTHRkMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAExEsr'.PHP_EOL.'/55aqgFXdrbRNz1/WSNI8UaSUxCka2kGEN1bXsJIzjkeyv12dRHo7H5OmY2/Z9sN'.PHP_EOL.'fgKhWj7elq0xSlcA0KNQME4wHQYDVR0OBBYEFKIGgCZoS388STT0qjoX/swKYBXh'.PHP_EOL.'MB8GA1UdIwQYMBaAFKIGgCZoS388STT0qjoX/swKYBXhMAwGA1UdEwQFMAMBAf8w'.PHP_EOL.'CgYIKoZIzj0EAwIDSQAwRgIhAK5OqQoBGR/pj2NOb+PyRKK4k4d3Muj9z/6LsJK+'.PHP_EOL.'kkgUAiEA+FY4SWKv4mfe0gsOBId0Aah/HtVZxDBe3bCXOQM8MMM='],
                ],
            ],
            [
                __DIR__.'/EC/DER/prime256v1-cert.der',
                [
                    'kty' => 'EC',
                    'crv' => 'P-256',
                    'x' => 'xEsr_55aqgFXdrbRNz1_WSNI8UaSUxCka2kGEN1bXsI',
                    'y' => 'SM45Hsr9dnUR6Ox-TpmNv2fbDX4CoVo-3patMUpXANA',
                    'x5t' => 'ZnnaQDssCKJQZLp6zyHssIZOa7o',
                    'x5t#256' => 'v7VlokKTGL3anRk8Nl0VcqVC9u5j2Fb5tdlQntUgDT4',
                    'x5c' => ['MIIB0jCCAXegAwIBAgIJAK2o1kQ5JwpUMAoGCCqGSM49BAMCMEUxCzAJBgNVBAYT'.PHP_EOL.'AkFVMRMwEQYDVQQIDApTb21lLVN0YXRlMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRn'.PHP_EOL.'aXRzIFB0eSBMdGQwHhcNMTUxMTA4MTUxMTU2WhcNMTYxMTA3MTUxMTU2WjBFMQsw'.PHP_EOL.'CQYDVQQGEwJBVTETMBEGA1UECAwKU29tZS1TdGF0ZTEhMB8GA1UECgwYSW50ZXJu'.PHP_EOL.'ZXQgV2lkZ2l0cyBQdHkgTHRkMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAExEsr'.PHP_EOL.'/55aqgFXdrbRNz1/WSNI8UaSUxCka2kGEN1bXsJIzjkeyv12dRHo7H5OmY2/Z9sN'.PHP_EOL.'fgKhWj7elq0xSlcA0KNQME4wHQYDVR0OBBYEFKIGgCZoS388STT0qjoX/swKYBXh'.PHP_EOL.'MB8GA1UdIwQYMBaAFKIGgCZoS388STT0qjoX/swKYBXhMAwGA1UdEwQFMAMBAf8w'.PHP_EOL.'CgYIKoZIzj0EAwIDSQAwRgIhAK5OqQoBGR/pj2NOb+PyRKK4k4d3Muj9z/6LsJK+'.PHP_EOL.'kkgUAiEA+FY4SWKv4mfe0gsOBId0Aah/HtVZxDBe3bCXOQM8MMM='],
                ],
            ],
        ];
    }

    /**
     * @test
     */
    public function loadX5CParameter()
    {
        $key = new JWK([
            'kty' => 'RSA',
            'use' => 'sig',
            'kid' => '1b94c',
            'n' => 'vrjOfz9Ccdgx5nQudyhdoR17V-IubWMeOZCwX_jj0hgAsz2J_pqYW08PLbK_PdiVGKPrqzmDIsLI7sA25VEnHU1uCLNwBuUiCO11_-7dYbsr4iJmG0Qu2j8DsVyT1azpJC_NG84Ty5KKthuCaPod7iI7w0LK9orSMhBEwwZDCxTWq4aYWAchc8t-emd9qOvWtVMDC2BXksRngh6X5bUYLy6AyHKvj-nUy1wgzjYQDwHMTplCoLtU-o-8SNnZ1tmRoGE9uJkBLdh5gFENabWnU5m1ZqZPdwS-qo-meMvVfJb6jJVWRpl2SUtCnYG2C32qvbWbjZ_jBPD5eunqsIo1vQ',
            'e' => 'AQAB',
            'x5c' => ['MIIDQjCCAiqgAwIBAgIGATz/FuLiMA0GCSqGSIb3DQEBBQUAMGIxCzAJB'.PHP_EOL.'gNVBAYTAlVTMQswCQYDVQQIEwJDTzEPMA0GA1UEBxMGRGVudmVyMRwwGgYD'.PHP_EOL.'VQQKExNQaW5nIElkZW50aXR5IENvcnAuMRcwFQYDVQQDEw5CcmlhbiBDYW1'.PHP_EOL.'wYmVsbDAeFw0xMzAyMjEyMzI5MTVaFw0xODA4MTQyMjI5MTVaMGIxCzAJBg'.PHP_EOL.'NVBAYTAlVTMQswCQYDVQQIEwJDTzEPMA0GA1UEBxMGRGVudmVyMRwwGgYDV'.PHP_EOL.'QQKExNQaW5nIElkZW50aXR5IENvcnAuMRcwFQYDVQQDEw5CcmlhbiBDYW1w'.PHP_EOL.'YmVsbDCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAL64zn8/QnH'.PHP_EOL.'YMeZ0LncoXaEde1fiLm1jHjmQsF/449IYALM9if6amFtPDy2yvz3YlRij66'.PHP_EOL.'s5gyLCyO7ANuVRJx1NbgizcAblIgjtdf/u3WG7K+IiZhtELto/A7Fck9Ws6'.PHP_EOL.'SQvzRvOE8uSirYbgmj6He4iO8NCyvaK0jIQRMMGQwsU1quGmFgHIXPLfnpn'.PHP_EOL.'fajr1rVTAwtgV5LEZ4Iel+W1GC8ugMhyr4/p1MtcIM42EA8BzE6ZQqC7VPq'.PHP_EOL.'PvEjZ2dbZkaBhPbiZAS3YeYBRDWm1p1OZtWamT3cEvqqPpnjL1XyW+oyVVk'.PHP_EOL.'aZdklLQp2Btgt9qr21m42f4wTw+Xrp6rCKNb0CAwEAATANBgkqhkiG9w0BA'.PHP_EOL.'QUFAAOCAQEAh8zGlfSlcI0o3rYDPBB07aXNswb4ECNIKG0CETTUxmXl9KUL'.PHP_EOL.'+9gGlqCz5iWLOgWsnrcKcY0vXPG9J1r9AqBNTqNgHq2G03X09266X5CpOe1'.PHP_EOL.'zFo+Owb1zxtp3PehFdfQJ610CDLEaS9V9Rqp17hCyybEpOGVwe8fnk+fbEL'.PHP_EOL.'2Bo3UPGrpsHzUoaGpDftmWssZkhpBJKVMJyf/RuP2SmmaIzmnw9JiSlYhzo'.PHP_EOL.'4tpzd5rFXhjRbg4zW9C+2qok+2+qDM1iJ684gPHMIY8aLWrdgQTxkumGmTq'.PHP_EOL.'gawR+N5MDtdPTEQ0XfIBc2cJEUyMTY5MPvACWpkA6SdS4xSvdXK3IVfOWA=='],
        ]);

        $certificate = JWKFactory::createFromX5C($key->get('x5c'), ['use' => 'sig', 'kid' => '1b94c']);

        static::assertEquals([
            'kty' => 'RSA',
            'use' => 'sig',
            'kid' => '1b94c',
            'n' => 'vrjOfz9Ccdgx5nQudyhdoR17V-IubWMeOZCwX_jj0hgAsz2J_pqYW08PLbK_PdiVGKPrqzmDIsLI7sA25VEnHU1uCLNwBuUiCO11_-7dYbsr4iJmG0Qu2j8DsVyT1azpJC_NG84Ty5KKthuCaPod7iI7w0LK9orSMhBEwwZDCxTWq4aYWAchc8t-emd9qOvWtVMDC2BXksRngh6X5bUYLy6AyHKvj-nUy1wgzjYQDwHMTplCoLtU-o-8SNnZ1tmRoGE9uJkBLdh5gFENabWnU5m1ZqZPdwS-qo-meMvVfJb6jJVWRpl2SUtCnYG2C32qvbWbjZ_jBPD5eunqsIo1vQ',
            'e' => 'AQAB',
            'x5t' => '4pNenEBLv0JpLIdugWxQkOsZcK0',
            'x5t#256' => 'pJm2BBpkB8y7tCqrWM0X37WOmQTO8zQw-VpxVgBb21I',
            'x5c' => ['MIIDQjCCAiqgAwIBAgIGATz/FuLiMA0GCSqGSIb3DQEBBQUAMGIxCzAJBgNVBAYT'.PHP_EOL.'AlVTMQswCQYDVQQIEwJDTzEPMA0GA1UEBxMGRGVudmVyMRwwGgYDVQQKExNQaW5n'.PHP_EOL.'IElkZW50aXR5IENvcnAuMRcwFQYDVQQDEw5CcmlhbiBDYW1wYmVsbDAeFw0xMzAy'.PHP_EOL.'MjEyMzI5MTVaFw0xODA4MTQyMjI5MTVaMGIxCzAJBgNVBAYTAlVTMQswCQYDVQQI'.PHP_EOL.'EwJDTzEPMA0GA1UEBxMGRGVudmVyMRwwGgYDVQQKExNQaW5nIElkZW50aXR5IENv'.PHP_EOL.'cnAuMRcwFQYDVQQDEw5CcmlhbiBDYW1wYmVsbDCCASIwDQYJKoZIhvcNAQEBBQAD'.PHP_EOL.'ggEPADCCAQoCggEBAL64zn8/QnHYMeZ0LncoXaEde1fiLm1jHjmQsF/449IYALM9'.PHP_EOL.'if6amFtPDy2yvz3YlRij66s5gyLCyO7ANuVRJx1NbgizcAblIgjtdf/u3WG7K+Ii'.PHP_EOL.'ZhtELto/A7Fck9Ws6SQvzRvOE8uSirYbgmj6He4iO8NCyvaK0jIQRMMGQwsU1quG'.PHP_EOL.'mFgHIXPLfnpnfajr1rVTAwtgV5LEZ4Iel+W1GC8ugMhyr4/p1MtcIM42EA8BzE6Z'.PHP_EOL.'QqC7VPqPvEjZ2dbZkaBhPbiZAS3YeYBRDWm1p1OZtWamT3cEvqqPpnjL1XyW+oyV'.PHP_EOL.'VkaZdklLQp2Btgt9qr21m42f4wTw+Xrp6rCKNb0CAwEAATANBgkqhkiG9w0BAQUF'.PHP_EOL.'AAOCAQEAh8zGlfSlcI0o3rYDPBB07aXNswb4ECNIKG0CETTUxmXl9KUL+9gGlqCz'.PHP_EOL.'5iWLOgWsnrcKcY0vXPG9J1r9AqBNTqNgHq2G03X09266X5CpOe1zFo+Owb1zxtp3'.PHP_EOL.'PehFdfQJ610CDLEaS9V9Rqp17hCyybEpOGVwe8fnk+fbEL2Bo3UPGrpsHzUoaGpD'.PHP_EOL.'ftmWssZkhpBJKVMJyf/RuP2SmmaIzmnw9JiSlYhzo4tpzd5rFXhjRbg4zW9C+2qo'.PHP_EOL.'k+2+qDM1iJ684gPHMIY8aLWrdgQTxkumGmTqgawR+N5MDtdPTEQ0XfIBc2cJEUyM'.PHP_EOL.'TY5MPvACWpkA6SdS4xSvdXK3IVfOWA=='],
        ],
            $certificate->all()
        );
    }
}
