<?php

/**
 * Класс для формирования HTML документа.
 */
class document {

    /**
     * @deprecated
     */
    private $title, $msg, $err, $act, $description, $keywords;
    protected
            $_title = "",
            $_tabs = array(),
            $_breadcrumbs = array(),
            $_actions = array(),
            $_errors = array(),
            $_messages = array(),
            $_informations = array(),
            $_description = "",
            $_keywords = array(),
            $_echo_content,
            $_is_displayed = false,
            $_system_styles = array(
                '/sys/themes/.common/font-awesome.min.css',
                '/sys/themes/.common/icons.css',
                '/sys/themes/.common/flag.min.css'
                    ),
            $_system_scripts = array();

    public function __construct($group = 0) {
        $this->_title = __(dcms::getInstance()->title);
        if ($group > current_user::getInstance()->group) {
            $this->showAccessDenied(__('Доступ к данной странице запрещен'));
        }
        ob_start();
    }

    /**
     * Получение заголовка страницы
     * @return string
     */
    public function getTitle() {
        return $this->_title;
    }

    /**
     * Установка заголовка страницы
     * @param string $str
     */
    public function setTitle($str) {
        $this->_title = $str;
    }

    /**
     * Отображение страницы с ошибкой
     * @param string $err Текст ошибки
     */
    public function showAccessDenied($err) {
        if (isset($_GET['return']) && $url_return = new url($_GET['return'])) {
            if ($url_return->isInternalLink()) {
                header('Refresh: 2; url=' . $_GET['return']);
            }
        }
        $this->addError($err);
        $this->_display();
        exit;
    }

    #---------------------- Tabs -----------------------------#

    /**
     * Наличие вкладок
     * @return bool
     */
    public function hasTabs() {
        return !!$this->_tabs;
    }

    /**
     * Добавление вкладки
     * @param string $name Отображаемое название вкладки
     * @param string|url $url Ссылка
     * @param bool $selected Вкладка является текущей
     * @return document_link
     */
    public function addTab($name, $url, $selected = false) {
        return $this->_tabs[] = new document_link(text::toValue($name), $url, $selected);
    }

    /**
     * вывод вкладок
     * @param string $pattern
     * @param string $selected_class
     */
    public function displayTabs($pattern = '<a href="{$url}" class="tab {$selected}">{$name}</a>', $selected_class = "selected") {
        foreach ($this->_tabs AS $tab) {
            $replace = array(
                '{$url}' => $tab->url,
                '{$name}' => $tab->name,
                '{$selected}' => $tab->selected ? $selected_class : ''
            );
            echo str_replace(array_keys($replace), array_values($replace), $pattern);
        }
    }

    #---------------------- Breadcrumbs -----------------------------#

    /**
     * Наличие хлебных крошек
     * @return bool
     */
    public function hasBreadcrumbs() {
        return !!$this->_breadcrumbs;
    }

    /**
     * Добавление "хлебной крошки" (путь назад к главной странице)
     * @param string $name Отображаемое название
     * @param string|url $url Ссылка
     * @return document_link
     */
    public function addBreadcrumb($name, $url) {
        return $this->_breadcrumbs[] = new document_link(text::toValue($name), $url);
    }

    /**
     * Наличие ссылок действий
     * @return bool
     */
    public function hasActions() {
        return !!$this->_actions;
    }

    /**
     * Добавление ссылки "Действие"
     * @param string $name Название
     * @param string|url $url Ссылка
     * @return document_link
     */
    public function addAction($name, $url) {
        return $this->_actions[] = new document_link(text::toValue($name), $url);
    }

    /**
     * вывод действий
     * @param string $pattern
     */
    public function displayActions($pattern = '<a href="{$url}" class="action">{$name}</a>') {
        foreach ($this->_actions AS $action) {
            $replace = array(
                '{$url}' => $action->url,
                '{$name}' => $action->name
            );
            echo str_replace(array_keys($replace), array_values($replace), $pattern);
        }
    }

    #---------------------- Errors -----------------------------#

    /**
     * Наличие ошибок
     * @return bool
     */
    public function hasErrors() {
        return !!$this->_errors;
    }

    /**
     * Добавление сообщения об ошибке
     * @param string $text Текст сообщения об ошибке
     * @return document_message
     */
    public function addError($text) {
        return $this->_errors[] = new document_message($text, true);
    }

    /**
     * вывод ошибок
     * @param string $pattern
     */
    public function displayErrors($pattern = '<div class="ui negative message">{$text}</div>') {
        foreach ($this->_errors AS $error) {
            echo str_replace('{$text}', $error->text, $pattern);
        }
    }

    #---------------------- Messages -----------------------------#

    /**
     * Наличие сообщений
     * @return bool
     */
    public function hasMessages() {
        return !!$this->_messages;
    }

    /**
     * Информационное сообщение (или сообщение об успехе)
     * @param string $text Текст сообщения
     * @return document_message
     */
    public function addMessage($text) {
        return $this->_messages[] = new document_message($text);
    }

    /**
     * вывод сообщений
     * @param string $pattern
     */
    public function displayMessages($pattern = '<div class="ui positive message">{$text}</div>') {
        foreach ($this->_messages AS $msg) {
            echo str_replace('{$text}', $msg->text, $pattern);
        }
    }

    #---------------------- Informations -----------------------------#

    /**
     * Наличие информационных сообщений
     * @return bool
     */
    public function hasInformations() {
        return !!$this->_informations;
    }

    /**
     * Информационное сообщение (или сообщение об успехе)
     * @param string $text Текст сообщения
     * @return document_message
     */
    public function addInformations($text) {
        return $this->_informations[] = new document_message($text);
    }

    /**
     * вывод информационных сообщений
     * @param string $pattern
     */
    public function displayInformations($pattern = '<div class="ui info message">{$text}</div>') {
        foreach ($this->_informations AS $info) {
            echo str_replace('{$text}', $info->text, $pattern);
        }
    }

    /**
     * Переадресация на адрес, указанный в GET параметре return или в $default_url
     * @param string $default_url
     * @param int $timeout Время, через которое произойдет переадресация
     */
    public function toReturn($default_url = '/', $timeout = 2) {
        if ($default_url instanceof url) {
            $url = $default_url->getUrl();
        } else {
            $url = $default_url;
        }

        if (!empty($_GET['return'])) {
            $url_return = new url($_GET['return']);
            if ($url_return->isInternalLink()) {
                $url = $url_return->getUrl();
            }
        }
        if ($timeout) {
            header('Refresh: ' . intval($timeout) . '; url=' . $url);
        } else {
            // если задержки быть не должно, то ничего на клиент не отправляем и работу скрипта прерываем
            header('Location: ' . $url);
            $this->_is_displayed = true;
            exit;
        }
    }

    /**
     * Отображение содержимого блока темы
     * @param string $section
     */
    public function displaySection($section) {
        $design = new design();
        if ($section === $design->theme->getEchoSectionKey()) {
            echo $this->_echo_content;
        }
        $widgets = $design->theme->getWidgets($section);
        foreach ($widgets as $widget_name) {
            $widget = new widget(H . '/sys/widgets/' . $widget_name); // открываем
            $widget->display(); // отображаем
        }
    }

    /**
     * Получение времени генерации страницы на текущий момент
     * @return float
     */
    public function getGenerationTime() {
        return round(microtime(true) - TIME_START, 3);
    }

    /**
     * Массив подключаемых стилей
     * @return string[]
     */
    public function getStyles() {
        $design = new design();
        return array_merge($this->_system_styles, $design->theme->getStyles());
    }

    /**
     * Массив подключаемых скриптов
     * @return string[]
     */
    public function getScripts() {
        $design = new design();
        return array_merge($this->_system_scripts, $design->theme->getScripts());
    }

    /**
     * вывод подключений стилей
     * @param string $pattern
     */
    public function displayStyleSheetsTags($pattern = '<link rel="stylesheet" href="{$path}" type="text/css"/>') {
        $styles = $this->getStyles();
        foreach ($styles AS $path) {
            echo str_replace('{$path}', $path, $pattern);
        }
    }

    /**
     * вывод подключений скриптов
     * @param string $pattern
     */
    public function displayScriptsTags($pattern = '<script charset="utf-8" src="{$path}" type="text/javascript"></script>') {
        $scripts = $this->getScripts();
        foreach ($scripts AS $path) {
            echo str_replace('{$path}', $path, $pattern);
        }
    }

    public function displayMetaKeywords($pattern = '<meta name="keywords" content="{$content}" />') {
        if ($this->hasKeywords()) {
            echo str_replace('{$content}', join(', ', $this->getKeywords()), $pattern);
        }
    }

    public function displayMetaDescription($pattern = '<meta name="description" content="{$content}" />') {
        if ($this->hasDescription()) {
            echo str_replace('{$content}', $this->getDescription(), $pattern);
        }
    }

    /**
     * вывод хлебных крошек
     * @param string $pattern
     * @param string $selected_class
     */
    public function displayBreadcrumbs($pattern = '<a href="{$url}" class="breadcrumb {$selected}">{$name}</a>', $selected_class = "selected") {
        foreach ($this->_breadcrumbs AS $bc) {
            $replace = array(
                '{$url}' => $bc->url,
                '{$name}' => $bc->name,
                '{$selected}' => $bc->selected ? $selected_class : ''
            );
            echo str_replace(array_keys($replace), array_values($replace), $pattern);
        }
    }

    /**
     * Очистка вывода
     * Тема оформления применяться не будет
     */
    public function clean() {
        $this->_is_displayed = true;
        ob_clean();
    }

    /**
     * Формирование HTML документа и отправка данных браузеру
     */
    protected function _display() {
        if ($this->_is_displayed) {
            // повторная отправка html кода вызовет нарушение синтаксиса документа, да и вообще нам этого нафиг не надо
            return;
        }
        $this->_is_displayed = true;
        header('Cache-Control: no-store, no-cache, must-revalidate', true);
        header('Expires: ' . date('r'), true);
        header('X-UA-Compatible: IE=edge', true); // отключение режима совместимости в осле
        header('Content-Type: text/html; charset=utf-8', true);

        $design = new design();
        $design->assign('model', $this);
        $design->assign('actions', $this->_actions); // ссылки к действию
        $design->assign('options', $this->_options); // ссылки для опций
        $design->assign('returns', $this->_returns); // ссылки для возврата
        $design->assign('tabs', $this->_tabs); // вкладки

        $this->_echo_content = ob_get_clean(); // то, что попало в буфер обмена при помощи echo (display())

        if (dcms::getInstance()->align_html) {
            // форматирование HTML кода
            $document_content = $design->fetch('document.tpl');
            $align = new alignedxhtml();
            echo $align->parse($document_content);
        } else {
            $design->display('document.tpl');
        }
    }

    /**
     * То что срабатывает при exit
     */
    public function __destruct() {
        $this->_display();
    }

    /**
     * Список вкладок
     * @return document_link[]
     */
    public function getTabs() {
        return $this->_tabs;
    }

    /**
     * массив хлебных крошек
     * @return document_link[]
     */
    public function getBreadcrumbs() {
        return $this->_breadcrumbs;
    }

    /**
     * Массив ссылок "Действие"
     * @return document_link[]
     */
    public function getActions() {
        return $this->_actions;
    }

    /**
     * Массив сообщений об ощибках
     * @return document_message[]
     */
    public function getErrors() {
        return $this->_errors;
    }

    /**
     * Массив сообщений
     * @return document_message[]
     */
    public function getMessages() {
        return $this->_messages;
    }

    /**
     * Массив сообщений
     * @return document_message[]
     */
    public function getInformations() {
        return $this->_informations;
    }

    /**
     * Описание страницы (meta description)
     * @return string
     */
    public function getDescription() {
        return $this->_description;
    }

    /**
     * Описание страницы (meta description)
     * @param string $description
     */
    public function setDescription($description) {
        $this->_description = $description;
    }

    /**
     * Наличие описания страницы
     * @return bool
     */
    public function hasDescription() {
        return !!$this->_description;
    }

    /**
     * Добавление ключевого слова (meta keywords)
     * @param $string
     */
    public function addKeyword($string) {
        $this->_keywords[] = $string;
    }

    /**
     * Массив ключевых слов (meta keywords)
     * @return string[]
     */
    public function getKeywords() {
        return $this->_keywords;
    }

    /**
     * Массив ключевых слов (meta keywords)
     * @param string[] $keywords
     */
    public function setKeywords($keywords) {
        $this->_keywords = $keywords;
    }

    /**
     * наличие ключевых слов
     * @return bool
     */
    public function hasKeywords() {
        return !!$this->_keywords;
    }

    //region Deprecated (Устаревшие методы)

    /**
     * @param $name
     * @param string|url $url
     * @param bool $selected
     * @deprecated
     * @return document_link
     */
    function tab($name, $url, $selected = false) {
        return $this->addTab($name, $url, $selected);
    }

    /**
     * @param $name
     * @param string|url $url
     * @deprecated
     * @return document_link
     */
    function ret($name, $url) {
        return $this->addBreadcrumb($name, $url);
    }

    /**
     * @param $name
     * @param string|url $url
     * @deprecated
     * @return document_link
     */
    function act($name, $url) {
        return $this->addAction($name, $url);
    }

    /**
     * @param $name
     * @param string|url $url
     * @deprecated
     * @return document_link
     */
    function opt($name, $url) {
        return $this->addAction($name, $url);
    }

    /**
     * @param $text
     * @deprecated
     * @return document_message
     */
    function err($text) {
        return $this->addError($text);
    }

    /**
     * @param $text
     * @deprecated
     * @return document_message
     */
    function msg($text) {
        return $this->addMessage($text);
    }

    /**
     * @param $text
     * @deprecated
     * @return document_message
     */
    function info($text) {
        return $this->addInformations($text);
    }

    /**
     * Отображение страницы с ошибкой
     * @param string $err Текст ошибки
     * @deprecated
     */
    function access_denied($err) {
        $this->showAccessDenied($err);
    }

    public function __get($name) {
        switch ($name) {
            case 'msg':
                return $this->getMessages();
            case 'info':
                return $this->getInformations();
            case 'err':
                return $this->getErrors();
            case 'keywords':
                return $this->getKeywords();
            case 'description':
                return $this->getDescription();
        }

        return null;
    }

    public function __set($name, $value) {
        switch ($name) {
            case 'title':
                $this->setTitle($value);
                break;
            case 'keywords':
                $this->setKeywords($value);
                break;
            case 'description':
                $this->setDescription($value);
                break;
        }
    }

    //endregion
}
